package com.yeejoin.amos.api.householdapi.Utils;

import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.github.pagehelper.PageInfo;
import com.yeejoin.amos.api.householdapi.constant.TanYinConstant;
import com.yeejoin.amos.api.householdapi.exception.BusinessException;
import com.yeejoin.amos.api.householdapi.face.dto.TanYinAccessTokenDTO;
import com.yeejoin.amos.api.householdapi.face.dto.TanYinBaseResultDTO;
import com.yeejoin.amos.api.householdapi.face.dto.TanYinPageResultDTO;
import com.yeejoin.amos.api.householdapi.face.orm.houseapi.entity.houseapi.HYGFThirdStationLog;
import com.yeejoin.amos.api.householdapi.face.service.IHYGFThirdStationLogService;
import com.yeejoin.amos.openapi.enums.PVProducerInfoEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 碳银接口类
 *
 * <p>
 * ProjectName: amos-boot-zx-biz
 * PackageName: com.yeejoin.amos.api.householdapi.Utils
 *
 * @author yangyang
 * @version v1.0
 * @date 2024/7/15 10:25
 */
@Slf4j
@Component
public class TanYinApiUtils {

    @Value ("${tanYin.api.apiUrl:}")
    private String apiUrl;

    @Value ("${tanYin.api.clientSecret:}")
    private String clientSecret;

    @Value ("${tanYin.api.clientKey:}")
    private String clientKey;

    @Resource (type = RedisUtils.class)
    private RedisUtils redisUtils;

    @Autowired
    private IHYGFThirdStationLogService thirdStationLogService;

    /**
     * 获取 accessToken 接口
     *
     * @param clientKey    用户 ID
     * @param clientSecret 用户 Secret
     * @return {@link  String}
     * @throws
     * @author yangyang
     * @date 2024/7/15 14:00
     */
    private String getAccessToken(String clientKey, String clientSecret) {
        String accessToken = (String) redisUtils.get(TanYinConstant.ACCESS_TOKEN_KEY);
        if (StringUtils.isNotEmpty(accessToken)) {
            return accessToken;
        }
        String url = apiUrl + TanYinConstant.getAccessTokenUrl + String.format("?clientSecret=%s&grantType=client_credentials&clientKey=%s", clientSecret, clientKey);
        HttpResponse response = HttpUtil.createGet(url).execute();
        TanYinAccessTokenDTO accessTokenDTO = parseResponse("获取 accessToken 接口", response, TanYinAccessTokenDTO.class);
        putAccessTokenInCache(accessTokenDTO);
        return accessTokenDTO.getAccessToken();
    }

    /**
     * 刷新 accessToken 接口
     *
     * @param clientKey    用户 ID
     * @param clientSecret 用户 Secret
     * @return {@link  String}
     * @throws
     * @author yangyang
     * @date 2024/7/15 13:59
     */
    private String refreshAccessToken(String clientKey, String clientSecret) {
        String url = apiUrl + TanYinConstant.refreshAccessTokenUrl + String.format("?clientSecret=%s&grantType=client_credentials&clientKey=%s", clientSecret, clientKey);
        HttpResponse response = HttpUtil.createGet(url).execute();
        TanYinAccessTokenDTO accessTokenDTO = parseResponse("刷新 accessToken 接口", response, TanYinAccessTokenDTO.class);
        putAccessTokenInCache(accessTokenDTO);
        return accessTokenDTO.getAccessToken();
    }

    private void putAccessTokenInCache(TanYinAccessTokenDTO accessTokenDTO) {
        redisUtils.set(TanYinConstant.ACCESS_TOKEN_KEY, accessTokenDTO.getAccessToken(), accessTokenDTO.getAccessTokenExpiresIn() != null && accessTokenDTO.getAccessTokenExpiresIn() > 2 ? accessTokenDTO.getAccessTokenExpiresIn() - 1 : 1);
    }

    /**
     * 使用GET方法向指定URL发送请求，并解析返回的数据。
     *
     * @param desc        对请求的描述，用于日志记录。
     * @param url         请求的URL地址。
     * @param resultClass 返回结果的类型，用于解析响应体。
     * @param <T>         请求结果的类型参数。
     * @param <R>         返回类型参数，与resultClass相同。
     * @return 解析后的返回结果，如果请求失败或URL为空，则返回null。
     */
    public <T, R> R get(String desc, String url, Class<T> resultClass) {
        // 检查URL是否为空，如果为空则直接返回null。
        if (StringUtils.isBlank(url)) {
            return null;
        }
        // 记录请求开始的日志信息。
        log.debug("请求 => 碳银{}接口，url：{}", desc, url);
        Map<String, String> headers = new HashMap<>();
        HttpResponse response =null;
        try {
            // 添加Authorization头，使用Bearer令牌模式。
            headers.put("Authorization", "Bearer " + getAccessToken(clientKey, clientSecret));
            // 发送GET请求并获取响应。
            response = HttpUtil.createGet(url).execute();
            //存储日志
            HYGFThirdStationLog hygfThirdStationLog = new HYGFThirdStationLog();
            hygfThirdStationLog.setThirdCode(PVProducerInfoEnum.TANYIN.getCode());
            hygfThirdStationLog.setReqMethod(HttpMethod.GET.name());
            hygfThirdStationLog.setReqHeaders(headers.toString());
            hygfThirdStationLog.setReqPath(url);
            hygfThirdStationLog.setReqBody(null);
            hygfThirdStationLog.setResBody(response.toString());
            thirdStationLogService.saveLog(hygfThirdStationLog);
            // 解析响应并返回结果。
            return parseResponse(desc, response, resultClass);
        } catch (BusinessException businessException) {
            // 在业务异常发生时，尝试刷新令牌并重新发送请求。
            // 刷新 Access_token
            headers.put("Authorization", "Bearer " + refreshAccessToken(clientKey, clientSecret));
            response = HttpUtil.createGet(url).execute();
            return parseResponse(desc, response, resultClass);
        } catch (Exception e) {
            //存储日志
            HYGFThirdStationLog hygfThirdStationLog = new HYGFThirdStationLog();
            hygfThirdStationLog.setThirdCode(PVProducerInfoEnum.TANYIN.getCode());
            hygfThirdStationLog.setReqMethod(HttpMethod.GET.name());
            hygfThirdStationLog.setReqHeaders(headers.toString());
            hygfThirdStationLog.setReqPath(url);
            hygfThirdStationLog.setReqBody(null);
            if(Objects.nonNull(response)){
                hygfThirdStationLog.setResBody(response.toString());
            }
            thirdStationLogService.saveLog(hygfThirdStationLog);
            // 记录未预期异常的日志信息。
            log.warn(String.format("异常 => 碳银%s接口", desc), e);
            return null;
        }
    }

    /**
     * 使用POST方法向指定URL发送带参数和头部信息的请求，并解析返回的结果。
     *
     * @param desc        请求描述，用于日志记录。
     * @param url         请求的URL地址。
     * @param params      请求的参数，以Map形式提供。
     * @param headers     请求的头部信息，以Map形式提供。
     * @param resultClass 返回结果的类类型，用于解析响应。
     * @param <T>         请求参数的泛型类型。
     * @param <R>         响应结果的泛型类型。
     * @return 解析后的响应结果，如果请求失败或解析异常，可能返回null。
     */
    public <T, R> R post(String desc, String url, Map<String, Object> params, Map<String, String> headers, Class<T> resultClass) {
        // 检查URL是否为空，如果为空直接返回null。
        if (StringUtils.isBlank(url)) {
            return null;
        }
        // 将参数转换为JSON字符串。
        String paramsJsonStr = JSON.toJSONString(params);

        // 初始化headers，默认添加Authorization头。
        if (headers == null) {
            headers = new HashMap<>();
        }
        headers.put("Authorization", "Bearer " + getAccessToken(clientKey, clientSecret));

        // 记录请求日志。
        log.info("请求 => 碳银{}接口，url：{}，参数：{}，headers：{}", desc, url, paramsJsonStr, JSON.toJSONString(headers));

        HttpResponse response = null;
        try {
            // 发送POST请求，带上参数和headers，并执行。
            response = HttpUtil.createPost(url).body(paramsJsonStr, MediaType.APPLICATION_JSON_UTF8_VALUE).headerMap(headers, true).execute();
            //存储日志
            HYGFThirdStationLog hygfThirdStationLog = new HYGFThirdStationLog();
            hygfThirdStationLog.setThirdCode(PVProducerInfoEnum.TANYIN.getCode());
            hygfThirdStationLog.setReqMethod(HttpMethod.POST.name());
            hygfThirdStationLog.setReqHeaders(headers.toString());
            hygfThirdStationLog.setReqPath(url);
            hygfThirdStationLog.setReqBody(paramsJsonStr);
            hygfThirdStationLog.setResBody(response.toString());
            thirdStationLogService.saveLog(hygfThirdStationLog);
            // 解析响应并返回。
            return parseResponse(desc, response, resultClass);
        } catch (BusinessException businessException) {
            if (businessException.isTokenError()) {
                // 在业务异常情况下，尝试刷新令牌并重试请求。
                // 刷新 Access_token
                headers.put("Authorization", "Bearer " + refreshAccessToken(clientKey, clientSecret));
                response = HttpUtil.createPost(url).body(paramsJsonStr, MediaType.APPLICATION_JSON_UTF8_VALUE).headerMap(headers, true).execute();
                // 解析刷新令牌后的响应并返回。
                return parseResponse(desc, response, resultClass);
            }
            log.warn(String.format("异常 => 碳银%s接口，参数: %s，响应: %s", desc, paramsJsonStr, response != null ? response.body() : businessException.getMessage()));
            return null;
        } catch (Exception e) {
            //存储日志
            HYGFThirdStationLog hygfThirdStationLog = new HYGFThirdStationLog();
            hygfThirdStationLog.setThirdCode(PVProducerInfoEnum.TANYIN.getCode());
            hygfThirdStationLog.setReqMethod(HttpMethod.GET.name());
            hygfThirdStationLog.setReqHeaders(headers.toString());
            hygfThirdStationLog.setReqPath(url);
            hygfThirdStationLog.setReqBody(null);
            if(Objects.nonNull(response)){
                hygfThirdStationLog.setResBody(response.toString());
            }
            thirdStationLogService.saveLog(hygfThirdStationLog);
            // 记录未预期的异常日志。
            log.warn(String.format("异常 => 碳银%s接口，参数: %s", desc, paramsJsonStr), e);
            return null;
        }
    }

    /**
     * 使用POST方法发送带分页信息的网络请求。
     *
     * @param desc          请求描述，用于日志记录。
     * @param url           请求的URL地址。
     * @param paramsJsonStr 请求参数的JSON字符串形式。
     * @param headers       请求的头部信息，可以为null。如果为null，则会添加默认的Authorization头。
     * @param resultClass   结果对象的类类型，用于解析响应。
     * @param <T>           结果对象的泛型类型。
     * @return 返回包含分页信息和请求结果的PageInfo对象，如果请求URL为空，则返回null。
     */
    public <T> PageInfo<T> postPage(String desc, String url, String paramsJsonStr, Map<String, String> headers, Class<T> resultClass) {
        // 检查URL是否为空，如果为空直接返回null
        if (StringUtils.isBlank(url)) {
            return null;
        }
        // 初始化headers，默认添加Authorization头。
        if (headers == null) {
            headers = new HashMap<>();
        }
        // 添加Authorization头，使用Bearer令牌模式
        headers.put("Authorization", "Bearer " + getAccessToken(clientKey, clientSecret));

        // 记录请求日志
        log.info("请求 => 碳银{}接口，url：{}，参数：{}，headers：{}", desc, url, paramsJsonStr, JSON.toJSONString(headers));

        HttpResponse response = null;
        try {
            // 发送POST请求，设置请求体和请求头
            response = HttpUtil.createPost(url).body(paramsJsonStr, MediaType.APPLICATION_JSON_UTF8_VALUE).headerMap(headers, true).execute();
            // 记录响应日志
            log.info("响应 => 碳银{}接口，参数：{}，httpCode：{}, response：{}", desc, paramsJsonStr, response.getStatus(), response.body());
            //存储日志
            HYGFThirdStationLog hygfThirdStationLog = new HYGFThirdStationLog();
            hygfThirdStationLog.setThirdCode(PVProducerInfoEnum.TANYIN.getCode());
            hygfThirdStationLog.setReqMethod(HttpMethod.POST.name());
            hygfThirdStationLog.setReqHeaders(headers.toString());
            hygfThirdStationLog.setReqPath(url);
            hygfThirdStationLog.setReqBody(paramsJsonStr);
            hygfThirdStationLog.setResBody(response.toString());
            thirdStationLogService.saveLog(hygfThirdStationLog);
            // 解析响应，返回分页信息和请求结果
            return parsePageResponse(desc, response, resultClass);
        } catch (BusinessException businessException) {
            if (businessException.isTokenError()) {
                // 在业务异常情况下，尝试刷新令牌并重试请求。
                // 刷新 Access_token
                headers.put("Authorization", "Bearer " + refreshAccessToken(clientKey, clientSecret));
                // 重新发送请求并解析响应
                response = HttpUtil.createPost(url).body(paramsJsonStr, MediaType.APPLICATION_JSON_UTF8_VALUE).headerMap(headers, true).execute();
                return parsePageResponse(desc, response, resultClass);
            }
            log.warn(String.format("异常 => 碳银%s接口，参数: %s，响应: %s", desc, paramsJsonStr, response != null ? response.body() : businessException.getMessage()));
            return null;
        } catch (Exception e) {
            //存储日志
            HYGFThirdStationLog hygfThirdStationLog = new HYGFThirdStationLog();
            hygfThirdStationLog.setThirdCode(PVProducerInfoEnum.TANYIN.getCode());
            hygfThirdStationLog.setReqMethod(HttpMethod.GET.name());
            hygfThirdStationLog.setReqHeaders(headers.toString());
            hygfThirdStationLog.setReqPath(url);
            hygfThirdStationLog.setReqBody(null);
            if(Objects.nonNull(response)){
                hygfThirdStationLog.setResBody(response.toString());
            }
            thirdStationLogService.saveLog(hygfThirdStationLog);
            // 记录异常日志
            log.warn(String.format("异常 => 碳银%s接口，参数: %s", desc, paramsJsonStr), e);
            return null;
        }
    }

    /**
     * 解析响应数据。
     * <p>
     * 该方法用于处理来自碳银接口的响应，解析响应体并根据预期的结果类型返回相应的数据。
     * 如果响应不成功或数据解析失败，将返回null或抛出异常。
     *
     * @param desc        接口描述，用于日志记录。
     * @param response    HTTP响应对象，包含响应状态码和响应体。
     * @param resultClass 预期结果的类类型，用于解析响应体并转换为相应的Java对象。
     * @param <T>         泛型参数，表示响应数据的类型。
     * @param <R>         泛型参数，表示返回数据的类型。
     * @return 解析后的结果对象，如果响应失败或数据无法解析，则返回null。
     */
    @SuppressWarnings ("unchecked")
    public <T, R> R parseResponse(String desc, HttpResponse response, Class<T> resultClass) {
        // 记录响应日志，包括接口描述、HTTP状态码和响应体。
        log.info("响应 => 碳银{}接口，httpCode：{}, response：{}", desc, response.getStatus(), response.body());

        // 检查响应是否成功，如果不成功则直接返回null。
        if (!response.isOk()) {
            return null;
        }

        // 解析响应体为TanYinBaseResultDTO对象，用于检查响应的通用状态。
        TanYinBaseResultDTO baseResult = JSONObject.parseObject(response.body(), TanYinBaseResultDTO.class);

        // 检查响应是否成功并存在有效数据，如果不成功且错误码为401，则抛出业务异常；其他情况返回null。
        if (!baseResult.getSuccess() || Objects.isNull(baseResult.getData())) {
            throw new BusinessException(baseResult.getCode().toString(), baseResult.getMsg());
        }

        // 解析响应数据部分，根据其类型进行相应的处理和转换。
        JSON result = (JSON) baseResult.getData();
        if (result instanceof Map) {
            // 如果结果是Map类型，则将其转换为指定的Java对象。
            return (R) JSON.parseObject(result.toJSONString()).toJavaObject(resultClass);
        }
        if (result instanceof List) {
            // 如果结果是List类型，则将其转换为指定的Java列表。
            return (R) JSON.parseArray(result.toJSONString()).toJavaList(resultClass);
        }

        // 如果结果类型既不是Map也不是List，则返回null。
        return null;
    }

    /**
     * 解析分页响应数据。
     * 该方法用于将请求的响应数据解析为特定类型的分页信息对象，方便后续处理和使用。
     *
     * @param desc        请求描述，用于日志记录，便于问题追踪。
     * @param response    HTTP响应对象，包含请求的返回状态码和响应体。
     * @param resultClass 结果对象的类类型，用于将JSON字符串转换为对应类型的对象。
     * @param <T>         泛型参数，表示结果对象的类型。
     * @return 返回解析后的分页信息对象，如果响应不成功或数据为空，则返回null。
     */
    public <T> PageInfo<T> parsePageResponse(String desc, HttpResponse response, Class<T> resultClass) {
        // 记录请求的响应信息到日志，包括描述、HTTP状态码和响应体。
         log.debug("响应 => 碳银{}接口，httpCode：{}, response：{}", desc, response.getStatus(), response.body());

        // 将响应体解析为TanYinBaseResultDTO对象，其中包含分页数据。
        TanYinBaseResultDTO<JSONObject> listResult = JSONObject.parseObject(response.body(), TanYinBaseResultDTO.class);

        // 检查响应是否成功且数据不为空，如果不满足则直接返回null。
        if (!response.isOk() || Objects.isNull(listResult.getData())) {
            return null;
        }

        if ("401".equals(listResult.getCode())) {
            // Token 过期
            throw new BusinessException(listResult.getCode().toString(), listResult.getMsg());
        }

        // 初始化PageInfo对象，用于存放解析后的分页数据。
        PageInfo<T> pageInfo = new PageInfo<>();

        // 从解析结果中获取具体的分页数据对象。
        TanYinPageResultDTO tanYinPageResultDTO = JSONObject.parseObject(listResult.getData().toJSONString(), TanYinPageResultDTO.class);

        // 将分页数据中的列表数据转换为指定类型的集合。
        if (tanYinPageResultDTO.getList() != null) {
            List<T> newList = (List<T>) tanYinPageResultDTO.getList().parallelStream().map(json -> JSON.toJavaObject((JSON) json, resultClass)).collect(Collectors.toList());
            // 将转换后的集合设置到PageInfo对象中。
            pageInfo.setList(newList);
        }

        // 设置分页信息，包括总页数、每页大小、当前页码和总记录数。
        pageInfo.setPages(tanYinPageResultDTO.getTotalPage() == null ? 0 : tanYinPageResultDTO.getTotalPage());
        pageInfo.setPageSize(tanYinPageResultDTO.getPageSize() == null ? 0 : tanYinPageResultDTO.getPageSize());
        pageInfo.setPageNum(tanYinPageResultDTO.getPageNo() == null ? 0 : tanYinPageResultDTO.getPageNo());
        pageInfo.setTotal(tanYinPageResultDTO.getTotal() == null ? 0 : tanYinPageResultDTO.getTotal());

        // 返回解析后的分页信息对象。
        return pageInfo;
    }

}
