package com.yeejoin.precontrol.controller;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yeejoin.precontrol.common.dto.CommonPageableDto;
import com.yeejoin.precontrol.common.dto.StudioPageableDto;
import com.yeejoin.precontrol.common.entity.Person;
import com.yeejoin.precontrol.common.entity.Reward;
import com.yeejoin.precontrol.common.service.IRewardService;
import com.yeejoin.precontrol.common.utils.FileHelper;
import com.yeejoin.precontrol.common.utils.Response;
import com.yeejoin.precontrol.common.utils.StringUtil;
import com.yeejoin.precontrol.common.vo.RewardAndRetributionListVo;
import com.yeejoin.precontrol.common.vo.RewardListVo;
import com.yeejoin.precontrol.common.vo.RewardOrgCodeVo;
import com.yeejoin.precontrol.common.vo.RewardVo;
import com.yeejoin.precontrol.controller.publics.BaseController;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;

/**
 * 奖励表
 *
 * @author duanwei
 * @date 2021-04-06
 */
@RestController
@Api(tags = "奖励表Api")
@RequestMapping(value = "/reward", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class RewardController extends BaseController {

	@Autowired
	IRewardService iRewardService;

	/**
	 * 新增奖励表
	 *
	 * @return
	 */
	@RequestMapping(value = "/save", method = RequestMethod.POST)
	@ApiOperation(httpMethod = "POST", value = "新增奖励表", notes = "新增奖励表")
	public boolean saveReward(HttpServletRequest request, @RequestBody Reward reward) {
		return iRewardService.save(reward);
	}

	/**
	 * 根据id删除
	 *
	 * @param id
	 * @return
	 */
	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ApiOperation(httpMethod = "DELETE", value = "根据id删除", notes = "根据id删除")
	public boolean deleteById(HttpServletRequest request, @PathVariable Long id) {
		return iRewardService.removeById(id);
	}

	/**
	 * 修改奖励表
	 *
	 * @return
	 */
	@RequestMapping(value = "/updateById", method = RequestMethod.PUT)
	@ApiOperation(httpMethod = "PUT", value = "修改奖励表", notes = "修改奖励表")
	public boolean updateByIdReward(HttpServletRequest request, @RequestBody Reward reward) {
		return iRewardService.updateById(reward);
	}

	/**
	 * 根据id查询
	 *
	 * @param id
	 * @return
	 */
	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ApiOperation(httpMethod = "GET", value = "根据id查询", notes = "根据id查询")
	public Reward selectById(HttpServletRequest request, @PathVariable Long id) {
		return iRewardService.getById(id);
	}

	/**
	 * 列表分页查询
	 *
	 * @return
	 */
	@RequestMapping(value = "/list", method = RequestMethod.GET)
	@ApiOperation(httpMethod = "GET", value = "列表分页查询", notes = "列表分页查询")
	public IPage<RewardVo> listPage(CommonPageableDto pageable, RewardVo rewardVo) {
		Person person = getPerson();
		if (person != null) {
			rewardVo.setPersonId(person.getId());
		}
		if (StringUtil.isNotEmpty(rewardVo.getTaskName())) {
			rewardVo.setTaskName("%" + rewardVo.getTaskName() + "%");
		}
		if (StringUtil.isNotEmpty(rewardVo.getTaskNo())) {
			rewardVo.setTaskNo("%" + rewardVo.getTaskNo() + "%");
		}
		return iRewardService.pageInfo(pageable, rewardVo);
	}


	/**
	 * 人员详情页面 查询该用户的检查违章数据
	 * @param pageable
	 * @param rewardVo
	 * @return
	 */
	@RequestMapping(value = "/list/page", method = RequestMethod.GET)
	@ApiOperation(httpMethod = "GET", value = "列表分页查询", notes = "列表分页查询")
	public Object pageByPersonId(CommonPageableDto pageable, RewardVo rewardVo) {
		Map map = new HashMap();
		BigDecimal totalSum = BigDecimal.ZERO;
		BigDecimal sendSum = BigDecimal.ZERO;
		if(StringUtils.isEmpty(rewardVo.getPersonId())){
			map.put("code","-1");
			map.put("message","没有数据");
			map.put("pageData", new Page<>(pageable.getPageNumber(), pageable.getPageSize()));
			map.put("totalSum", totalSum);
			map.put("sendSum", sendSum);
			return map;
		}
		IPage<RewardVo> rewardVoPage = iRewardService.pageInfo(pageable, rewardVo);
		Map ret  = iRewardService.sumMoneyTotalByPersonId(rewardVo.getPersonId());
		map.put("totalSum", ret.get("totalSum"));
		map.put("sendSum", ret.get("sendSum"));
		map.put("code","0");
		map.put("message","查询成功");
		map.put("pageData", rewardVoPage);
		return map;
	}

	/**
	 * 查处违章总数查询
	 *
	 * @return
	 */
	@RequestMapping(value = "/total", method = RequestMethod.GET)
	@ApiOperation(httpMethod = "GET", value = "查处违章总数查询", notes = "查处违章总数查询")
	public Object listTotal() {
		Response response = new Response();
		Map ret = new HashMap();
		Person person = getPerson();
		if (person != null) {
			RewardVo rewardVo = new RewardVo();
			rewardVo.setPersonId(person.getId());
			List<RewardVo> list = iRewardService.rewardAllList(rewardVo);
			ret.put("total", list.size());
			BigDecimal sumMoney = BigDecimal.ZERO;
			if (null != list && list.size() > 0) {
				sumMoney = list.stream().map(RewardVo::getMoney).reduce(BigDecimal.ZERO, BigDecimal::add);
			}
			ret.put("sumMoney", sumMoney.toPlainString());
		} else {
			ret.put("sumMoney", Integer.MIN_VALUE);
			ret.put("total", Integer.MIN_VALUE);
		}
		response.setData(ret);
		return response;
	}

	/**
	 * 根据人员id查询该人员的奖励违章列表
	 * 
	 * @throws UnsupportedEncodingException
	 */
	@GetMapping("/list/by/personId")
	@ApiOperation(httpMethod = "GET", value = "根据人员id查询该人员的奖励违章列表", notes = "列表分页查询")
	public IPage<RewardVo> listPageByUid(CommonPageableDto pageable, RewardVo rewardVo, HttpServletResponse response)
			throws UnsupportedEncodingException {
		String fileName = java.net.URLEncoder.encode("奖励信息表", "UTF-8") + ".xls";
		IPage<RewardVo> page = iRewardService.listPageByUid(pageable, rewardVo);
		if (rewardVo != null && rewardVo.getOperType() == 1) {
			List<RewardVo> excelList = page.getRecords();
			if (StringUtils.isEmpty(rewardVo.getBeforeCreateDate())
					&& StringUtils.isEmpty(rewardVo.getEndCreateDate())) {
				excelList = new ArrayList<RewardVo>();
			}
			FileHelper.exportExcel(excelList, "奖励信息", "奖励信息", RewardVo.class, fileName, response);
			return null;
		} else {
			return page;
		}
	}

	/**
	 * 根据人员id查询该人员的奖励违章数量统计
	 */
	@PostMapping(value = "/exchange/by/personId")
	@ApiOperation(httpMethod = "GET", value = "根据人员id查询该人员的奖励违章数量统计", notes = "根据人员id查询该人员的奖励违章数量统计")
	public RewardVo getRewardExchangeCount(@RequestBody RewardVo rewardVo) {
		return iRewardService.getRewardExchangeCount(rewardVo);
	}

	/**
	 * 根据orgcode查询该orgcode下的人员奖励违章列表
	 */
	@GetMapping("/list/user")
	@ApiOperation(httpMethod = "GET", value = "根据orgcode查询该orgcode下的人员奖励列表", notes = "根据orgcode查询该orgcode下的人员奖励列表")
	public IPage<RewardOrgCodeVo> listPageByOrgCode(StudioPageableDto pageable, RewardOrgCodeVo rewardVo) {
		Person person = getPerson();
		String orgCode = person != null ? person.getOrgCode() : null;
		return iRewardService.listPageByOrgCode(pageable, rewardVo, orgCode);
	}

	/**
	 * 根据人员id查询该人员的奖励列表
	 */
	@GetMapping("/list/user/detail")
	@ApiOperation(httpMethod = "GET", value = "根据人员id查询该人员的奖励详情列表", notes = "根据人员id查询该人员的奖励详情列表")
	public IPage<RewardListVo> rewardUserListDetail(StudioPageableDto pageable, RewardListVo rewardVo) {
		return iRewardService.rewardUserListDetail(pageable, rewardVo);
	}

	/**
	 * 奖励惩罚 列表
	 * 
	 * @param pageable
	 * @param rewardVo
	 * @return
	 */
	@GetMapping("/retribution/list")
	@ApiOperation(httpMethod = "GET", value = "奖励惩罚 列表", notes = "奖励惩罚 列表")
	public IPage<RewardAndRetributionListVo> rewardAndRetributionList(StudioPageableDto pageable,
			RewardAndRetributionListVo rewardVo) {
		return iRewardService.rewardAndRetributionList(pageable, rewardVo);
	}

	/**
	 * 奖励惩罚金额统计
	 * 
	 * @param type retributionAllTotal-所有惩罚总金额 retributionDayTotal-今日惩罚总金额
	 *             rewardAllTotal-所有奖励总金额 rewardDayTotal-今日奖励总金额
	 * @return
	 */
	@GetMapping("/report/total/money")
	@ApiOperation(httpMethod = "GET", value = "奖励惩罚金额统计", notes = "奖励惩罚金额统计")
	@ApiImplicitParam(name = "type", example = "retributionAllTotal-所有惩罚总金额  retributionDayTotal-今日惩罚总金额  rewardAllTotal-所有奖励总金额   rewardDayTotal-今日奖励总金额")
	public Object reportTotalMoney(@RequestParam("type") String type) {
		if (StringUtils.isEmpty(type)) {
			return "参数为空";
		}
		return iRewardService.reportTotalMoney(type);
	}

	/**
	 * 某个用户奖励 列表分页查询
	 *
	 * @return
	 */
	@RequestMapping(value = "/person/list", method = RequestMethod.GET)
	@ApiOperation(httpMethod = "GET", value = "某个用户奖励 列表分页查询", notes = "某个用户奖励 列表分页查询")
	public IPage<RewardVo> listPageByPersonId(CommonPageableDto pageable, RewardVo rewardVo) {
		if (StringUtils.isEmpty(rewardVo.getPersonId())) {
			return new Page<>(pageable.getPageNumber(), pageable.getPageSize());
		}
		return iRewardService.pageUserInfo(pageable, rewardVo);
	}

	/**
	 * 查处违章排名列表
	 *
	 * @return
	 */
	@RequestMapping(value = "/list/user/bad", method = RequestMethod.GET)
	@ApiOperation(httpMethod = "GET", value = " 查处违章排名列表", notes = " 查处违章排名列表")
	public IPage<RewardOrgCodeVo> listBadPageByPersonId(StudioPageableDto pageable, RewardOrgCodeVo rewardVo) {
		Person person = getPerson();
		String orgCode = person != null ? person.getOrgCode() : null;
		return iRewardService.listBadPageByOrgCode(pageable, rewardVo, null);
	}

}
