package com.yeejoin.amos.boot.module.hygf.api.util;


import com.yeejoin.amos.component.robot.BadRequest;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

public class FileUtils {
    public byte[] getContent(String filePath) throws IOException {
        File file = new File(filePath);
        long fileSize = file.length();
        if (fileSize > Integer.MAX_VALUE) {
            System.out.println("file too big...");
            return null;
        }
        FileInputStream fi = new FileInputStream(file);
        byte[] buffer = new byte[(int) fileSize];
        int offset = 0;
        int numRead = 0;
        while (offset < buffer.length
                && (numRead = fi.read(buffer, offset, buffer.length - offset)) >= 0) {
            offset += numRead;
        }
        // 确保所有数据均被读取
        if (offset != buffer.length) {
            throw new IOException("Could not completely read file "
                    + file.getName());
        }
        fi.close();
        return buffer;
    }


    /**
     * 创建ZIP文件
     *
     * @param filesToZip 要打包的文件列表
     * @param zipPath    ZIP文件保存路径
     * @throws IOException 如果I/O操作失败
     */
    public static void createZipFile(List<Path> filesToZip, String zipPath) throws IOException {
        try (FileOutputStream fos = new FileOutputStream(zipPath);
             ZipOutputStream zos = new ZipOutputStream(fos)) {
            for (Path file : filesToZip) {
                FileInputStream fis = new FileInputStream(file.toFile());
                ZipEntry zipEntry = new ZipEntry(file.getFileName().toString());
                zos.putNextEntry(zipEntry);

                byte[] bytes = new byte[1024];
                int length;
                while ((length = fis.read(bytes)) >= 0) {
                    zos.write(bytes, 0, length);
                }
                zos.closeEntry();
                fis.close();
            }
        }
    }

    /**
     *
     * 创建压缩包文件
    **/
    public static void addFileToZip(String filePath, String zipFileName, ZipOutputStream zos) throws IOException {
        File file = new File(filePath);
        FileInputStream fis = new FileInputStream(file);
        ZipEntry zipEntry = new ZipEntry(zipFileName);
        zos.putNextEntry(zipEntry);

        byte[] bytes = new byte[1024];
        int length;
        while ((length = fis.read(bytes)) >= 0) {
            zos.write(bytes, 0, length);
        }
        zos.closeEntry();
        fis.close();
    }

    public static void cleanup(Path tempDir) {
        try (Stream<Path> paths = Files.walk(tempDir)) {
            List<Path> filesToDelete = paths.sorted(Comparator.reverseOrder())
                    .collect(Collectors.toList());

            // 分批删除文件
            int batchSize = 100; // 根据实际情况调整批次大小
            for (int i = 0; i < filesToDelete.size(); i += batchSize) {
                List<Path> batch = filesToDelete.subList(i, Math.min(i + batchSize, filesToDelete.size()));
                batch.forEach(file -> {
                    try {
                        Files.delete(file);
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                });
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * the traditional io way
     *
     * @param filename
     * @return
     * @throws IOException
     */
    public static byte[] toByteArray(String filename) throws IOException {

        File f = new File(filename);
        if (!f.exists()) {
            throw new FileNotFoundException(filename);
        }

        ByteArrayOutputStream bos = new ByteArrayOutputStream((int) f.length());
        BufferedInputStream in = null;
        try {
            in = new BufferedInputStream(new FileInputStream(f));
            int buf_size = 1024;
            byte[] buffer = new byte[buf_size];
            int len = 0;
            while (-1 != (len = in.read(buffer, 0, buf_size))) {
                bos.write(buffer, 0, len);
            }
            return bos.toByteArray();
        } catch (IOException e) {
            e.printStackTrace();
            throw e;
        } finally {
            try {
                in.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            bos.close();
        }
    }

    /**
     * NIO way
     *
     * @param filename
     * @return
     * @throws IOException
     */
    public static byte[] toByteArray2(String filename) throws IOException {

        File f = new File(filename);
        if (!f.exists()) {
            throw new FileNotFoundException(filename);
        }

        FileChannel channel = null;
        FileInputStream fs = null;
        try {
            fs = new FileInputStream(f);
            channel = fs.getChannel();
            ByteBuffer byteBuffer = ByteBuffer.allocate((int) channel.size());
            while ((channel.read(byteBuffer)) > 0) {
                // do nothing
                // System.out.println("reading");
            }
            return byteBuffer.array();
        } catch (IOException e) {
            e.printStackTrace();
            throw e;
        } finally {
            try {
                channel.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
            try {
                fs.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Mapped File way MappedByteBuffer 可以在处理大文件时，提升性能
     *
     * @param filename
     * @return
     * @throws IOException
     */
    public static byte[] toByteArray3(String filename) throws IOException {

        FileChannel fc = null;
        try {
            fc = new RandomAccessFile(filename, "r").getChannel();
            MappedByteBuffer byteBuffer = fc.map(MapMode.READ_ONLY, 0,
                    fc.size()).load();
            System.out.println(byteBuffer.isLoaded());
            byte[] result = new byte[(int) fc.size()];
            if (byteBuffer.remaining() > 0) {
                // System.out.println("remain");
                byteBuffer.get(result, 0, byteBuffer.remaining());
            }
            return result;
        } catch (IOException e) {
            e.printStackTrace();
            throw e;
        } finally {
            try {
                fc.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public static boolean save2File(String fname, byte[] msg){
        OutputStream fos = null;
        try{
            File file = new File(fname);
            File parent = file.getParentFile();
            boolean bool;
            if ((!parent.exists()) &&
                    (!parent.mkdirs())) {
                return false;
            }
            fos = new FileOutputStream(file);
            fos.write(msg);
            fos.flush();
            return true;
        }catch (FileNotFoundException e){
            return false;
        }catch (IOException e){
            File parent;
            return false;
        }
        finally{
            if (fos != null) {
                try{
                    fos.close();
                }catch (IOException e) {}
            }
        }
    }

    public static void createZipFile(Path folderPath, String zipFilePath, HttpServletResponse response) {
        try (ServletOutputStream out = response.getOutputStream();
             ZipOutputStream zos = new ZipOutputStream(out)) {

            addFolderToZip(folderPath.toFile(), "", zos);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void addFolderToZip(File folder, String parentFolder, ZipOutputStream zos) throws IOException {
        File[] files = folder.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    addFolderToZip(file, parentFolder + file.getName() + "/", zos);
                } else {
                    addFileToZip(file, parentFolder, zos);
                }
            }
        }
    }

    private static void addFileToZip(File sourceFile, String parentFolder, ZipOutputStream zipOut) throws IOException {
        FileInputStream fin = new FileInputStream(sourceFile);
        String zipEntryName = parentFolder + sourceFile.getName();
        zipOut.putNextEntry(new ZipEntry(zipEntryName));

        byte[] buffer = new byte[1024];
        int length;
        while ((length = fin.read(buffer)) >= 0) {
            zipOut.write(buffer, 0, length);
        }
        zipOut.closeEntry();
        fin.close();
    }


    public static MultipartFile convertZipToMultipartFile(File zipFile) throws IOException {
        try (FileInputStream input = new FileInputStream(zipFile)) {
            return new MockMultipartFile(
                    zipFile.getName(), // name
                    zipFile.getName(), // originalFilename
                    "application/zip", // contentType
                    FileCopyUtils.copyToByteArray(input) // bytes
            );
        }
    }


    public static boolean unzipAndUpload( String localZipFilePath,HttpServletResponse response) throws Exception {
        boolean success = false;
        File tempDir = null;

        try {
            // 创建临时目录用于解压
            tempDir = Files.createTempDirectory("unzip_temp_").toFile();
            tempDir.deleteOnExit(); // 确保 JVM 退出时删除临时目录

            // 解压本地 ZIP 文件到临时目录
            unzipLocalFile(localZipFilePath, tempDir.getAbsolutePath());

            // 获取解压后的文件列表
            File[] files = tempDir.listFiles();
            if (files == null || files.length == 0) {
                throw new IOException("No files found after unzipping.");
            }


            createZipFile(tempDir.toPath(),tempDir.getPath(),response);


            success = true;
        } catch (Exception e) {

            throw e;
        } finally {
            // 清理临时文件和断开连接
            if (tempDir != null) {
                com.yeejoin.amos.boot.module.hygf.api.util.FileUtils.cleanup(tempDir.toPath());
            }

        }

        return success;
    }


    private static void unzipLocalFile(String zipFilePath, String destDir) throws IOException {
        byte[] buffer = new byte[1024];
        Path zipFilePathObj = Paths.get(zipFilePath);
        Path destDirObj = Paths.get(destDir);

        try (ZipInputStream zis = new ZipInputStream(new FileInputStream(zipFilePathObj.toFile()))) {
            ZipEntry zipEntry = zis.getNextEntry();

            while (zipEntry != null) {
                Path newFilePath = destDirObj.resolve(zipEntry.getName());

                if (!zipEntry.isDirectory()) {
                    // 如果是文件，则创建新文件并写入内容
                    Files.createDirectories(newFilePath.getParent());
                    try (FileOutputStream fos = new FileOutputStream(newFilePath.toFile())) {
                        int len;
                        while ((len = zis.read(buffer)) > 0) {
                            fos.write(buffer, 0, len);
                        }
                    }
                } else {
                    // 如果是目录，则创建目录
                    Files.createDirectories(newFilePath);
                }

                zipEntry = zis.getNextEntry();
            }
            zis.closeEntry();
        }
    }
}