package com.yeejoin.amos.knowledgebase.controller;

import com.yeejoin.amos.knowledgebase.face.model.KnowledgeTagModel;
import com.yeejoin.amos.knowledgebase.face.service.TagService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.typroject.tyboot.core.foundation.enumeration.UserType;
import org.typroject.tyboot.core.foundation.utils.StringUtil;
import org.typroject.tyboot.core.foundation.utils.ValidationUtil;
import org.typroject.tyboot.core.restful.doc.TycloudOperation;
import org.typroject.tyboot.core.restful.doc.TycloudResource;
import org.typroject.tyboot.core.restful.exception.instance.BadRequest;
import org.typroject.tyboot.core.restful.utils.ResponseHelper;
import org.typroject.tyboot.core.restful.utils.ResponseModel;

/**
 * <p>
 * 标签库 前端控制器
 * </p>
 *
 * @author 子杨
 * @since 2020-08-05
 */

@RestController
@TycloudResource(module = "knowledgebase", value = "tag")
@RequestMapping(value = "/v1/tag")
@Api(tags = "knowledgebase-标签库")
public class TagResource {

    private final Logger logger = LogManager.getLogger(TagResource.class);
    @Autowired
    private TagService tagService;


    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "创建标签")
    @RequestMapping(value = "", method = RequestMethod.POST)
    public ResponseModel<KnowledgeTagModel> create(@RequestBody KnowledgeTagModel model) {
        if (ValidationUtil.isEmpty(model)
                || ValidationUtil.isEmpty(model.getTagName())
                || ValidationUtil.isEmpty(model.getTagGroup())
                || ValidationUtil.isEmpty(model.getTagType())) {
            throw new BadRequest("参数校验失败.");
        }

        return ResponseHelper.buildResponse(tagService.createTag(model));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "更新标签")
    @RequestMapping(value = "/{sequenceNbr}", method = RequestMethod.PUT)
    public ResponseModel<KnowledgeTagModel> update(
            @RequestBody KnowledgeTagModel model,
            @PathVariable(value = "sequenceNbr") Long sequenceNbr) {
        if (ValidationUtil.isEmpty(model)
                || ValidationUtil.isEmpty(model.getTagName())
                || ValidationUtil.isEmpty(model.getTagGroup())) {
            throw new BadRequest("参数校验失败.");
        }

        model.setSequenceNbr(sequenceNbr);
        return ResponseHelper.buildResponse(tagService.updateTag(model));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询单个对象")
    @RequestMapping(value = "/{sequenceNbr}", method = RequestMethod.GET)
    public ResponseModel seleteOne(@PathVariable(value="sequenceNbr") Long sequenceNbr) {
        return ResponseHelper.buildResponse(tagService.detailTag(sequenceNbr));
    }


    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "条件查询")
    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public ResponseModel queryMapList(
            @RequestParam(value = "groupSeq", required = false) Long groupSeq,
            @RequestParam(value = "tagName", required = false) String tagName,
            @RequestParam(value = "tagCode", required = false) String tagCode,
            @RequestParam(value = "tagStatus", required = false) String tagStatus) {

        return ResponseHelper.buildResponse(tagService.queryTagMap(groupSeq, tagName, tagCode, tagStatus));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "按名称搜索")
    @RequestMapping(value = "/list4doc", method = RequestMethod.GET)
    public ResponseModel queryListForDoc(
            @RequestParam(value = "tagName", required = false) String tagName) {
        return ResponseHelper.buildResponse(tagService.queryTagDetailList(tagName));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询TOP15")
    @RequestMapping(value = "/list/top", method = RequestMethod.GET)
    public ResponseModel selectForList(@RequestParam(value = "quoteType", required = false) String quoteType) {
        boolean isAll = ValidationUtil.equalsIgnoreCase(quoteType, TagService.APPKEY_ALL);
        return ResponseHelper.buildResponse(tagService.queryTopList(isAll));
    }


    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "批量删除")
    @RequestMapping(value = "/{ids}", method = RequestMethod.DELETE)
    public ResponseModel batchOperate(@PathVariable(value = "ids") String ids) {
        return ResponseHelper.buildResponse(tagService.delete(StringUtil.String2LongList(ids)));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "启用标签")
    @RequestMapping(value = "status/activate/{ids}", method = RequestMethod.PUT)
    public ResponseModel batchActivate(@PathVariable(value = "ids") String ids) {
        return ResponseHelper.buildResponse(tagService.updateTagStatus(StringUtil.String2LongList(ids), TagService.TAG_STATUS_ACTIVATE));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "停用标签")
    @RequestMapping(value = "status/deactivate/{ids}", method = RequestMethod.PUT)
    public ResponseModel batchDeactivate(@PathVariable(value = "ids") String ids) {
        return ResponseHelper.buildResponse(tagService.updateTagStatus(StringUtil.String2LongList(ids), TagService.TAG_STATUS_DEACTIVATE));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "检查标签名称是否已存在")
    @RequestMapping(value = "/name/exist", method = RequestMethod.GET)
    public ResponseModel checkTagName(@RequestParam("tagName") String tagName,
                                      @RequestParam(value = "sequenceNbr", required = false) Long sequenceNbr) {
        if (ValidationUtil.isEmpty(tagName)) {
            throw new BadRequest("参数校验失败");
        }
        return ResponseHelper.buildResponse(tagService.tagNameIsExist(tagName, sequenceNbr));
    }

    @TycloudOperation(ApiLevel = UserType.AGENCY)
    @ApiOperation(value = "查询标签关联的战例列表")
    @RequestMapping(value = "/{sequenceNbr}/related/doc", method = RequestMethod.GET)
    public ResponseModel queryRelatedDocList(@PathVariable(value = "sequenceNbr") Long sequenceNbr) {
        return ResponseHelper.buildResponse(tagService.queryRelatedDocList(sequenceNbr));
    }
}
