package com.yeejoin.precontrol.common.utils;

import com.yeejoin.precontrol.common.exception.CommonException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;
import org.docx4j.model.datastorage.migration.AbstractMigratorUsingAnswersFormat;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.PropertySource;
import org.springframework.stereotype.Component;

import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author DELL
 */
@Component
@PropertySource("classpath:sms.properties")
public class MsgCodeUtil {

    @Value("${send.sms.v.1}")
    private String SEND_SMS_V_1;

    @Value("${sms.app.key}")
    private String APP_KEY;

    @Value("${sms.app.secret}")
    private String APP_SECRET;

    @Value("${sms.sender.verify}")
    private String SENDER;

    @Value("${sms.sender.notice}")
    private String SENDER_NOTICE;

    @Value("${sms.verify.template.id}")
    private String TEMPLATE_ID;

    @Value("${sms.signature}")
    private String SIGNATURE;

    @Value("${sms.reject.template.id}")
    private String TEMPLATE_REJECT_ID;

    //无需修改,用于格式化鉴权头域,给"X-WSSE"参数赋值
    private static final String WSSE_HEADER_FORMAT = "UsernameToken Username=\"%s\",PasswordDigest=\"%s\",Nonce=\"%s\",Created=\"%s\"";
    //无需修改,用于格式化鉴权头域,给"Authorization"参数赋值
    private static final String AUTH_HEADER_VALUE = "WSSE realm=\"SDP\",profile=\"UsernameToken\",type=\"Appkey\"";

    private Pattern p = Pattern.compile("^((13[0-9])|(14[0-9])|(15[0-9])|(16[0-9])|(17[0-9])|(18[0-9])|(19[0-9]))\\d{8}$");
    /**
     * 生成数字验证码
     */
    public String createRandomVcode(int verifyCodeLength) {
        int MAX = 9;
        int MIN = 1;
        ThreadLocalRandom random = ThreadLocalRandom.current();
        StringBuilder vcode = new StringBuilder();
        for (int i = 0; i < verifyCodeLength; i++) {
            vcode.append(random.nextInt(MAX - MIN + 1) + MIN);
        }
        return vcode.toString();
    }

    /**
     * 校验手机号是否合法
     */
    public Boolean isMobile(String number) {
        boolean flag = false;
        try {
            Matcher m = p.matcher(number);
            flag = m.matches();
        } catch (Exception e) {
            flag = false;
        }
        return flag;
    }

    /**
     * 验证码发送
     *
     * @param receiver   接受者
     * @param verifyCode 验证码
     * @return String
     * @throws Exception 异常
     */
    public String verifyMsgSend(String receiver, String verifyCode) throws Exception {
        //必填,全局号码格式(包含国家码),示例:+8615123456789,多个号码之间用英文逗号分隔
        /**
         * 选填,使用无变量模板时请赋空值 String templateParas = "";
         * 单变量模板示例:模板内容为"您的验证码是${1}"时,templateParas可填写为"[\"369751\"]"
         * 双变量模板示例:模板内容为"您有${1}件快递请到${2}领取"时,templateParas可填写为"[\"3\",\"人民公园正门\"]"
         * 模板中的每个变量都必须赋值，且取值不能为空
         * 查看更多模板和变量规范:产品介绍>模板和变量规范
         */
        //模板变量，此处以单变量验证码短信为例，请客户自行生成6位验证码，并定义为字符串类型，以杜绝首位0丢失的问题（例如：002569变成了2569）。
        String templateParas = "[" + verifyCode + "]";
        //请求Body,不携带签名名称时,signature请填null
        return msgSend(SENDER, receiver, templateParas, TEMPLATE_ID);
    }

    /**
     * 审核拒绝消息发送
     *
     * @param receiver 接受者
     * @return String
     * @throws Exception 异常
     */
    public String checkRejectMsgSend(String receiver) throws Exception {
        return this.msgSend(SENDER_NOTICE, receiver, "", TEMPLATE_REJECT_ID);
    }

    public static void main(String[] args) {

    }

    private String msgSend(String sender, String receiver, String templateParas, String templateId) throws Exception {
        //选填,短信状态报告接收地址,推荐使用域名,为空或者不填表示不接收状态报告
        String statusCallBack = "";
        String body = buildRequestBody(sender, receiver, templateId, templateParas, statusCallBack, SIGNATURE);
        if (null == body || body.isEmpty()) {
            throw new CommonException(700001, "短信内容为空");
        }
        //请求Headers中的X-WSSE参数值
        String wsseHeader = buildWsseHeader(APP_KEY, APP_SECRET);
        if (null == wsseHeader || wsseHeader.isEmpty()) {
            throw new CommonException(700001, "wsse header is null");
        }
        //如果JDK版本是1.8,可使用如下代码
        CloseableHttpClient client = HttpClients.custom()
                .setSSLContext(new SSLContextBuilder().loadTrustMaterial(null,
                        (x509CertChain, authType) -> true).build())
                .setSSLHostnameVerifier(new DefaultHostnameVerifier())
                .build();

        HttpResponse response = client.execute(RequestBuilder.create("POST")//请求方法POST
                .setUri(SEND_SMS_V_1)
                .addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded")
                .addHeader(HttpHeaders.AUTHORIZATION, AUTH_HEADER_VALUE)
                .addHeader("X-WSSE", wsseHeader)
                .setEntity(new StringEntity(body)).build());
        System.out.println(response.toString()); //打印响应头域信息
        System.out.println(EntityUtils.toString(response.getEntity())); //打印响应消息实体
        return body;
    }

    /**
     * 构造请求Body体
     *
     * @param sender
     * @param receiver
     * @param templateId
     * @param templateParas
     * @param statusCallbackUrl
     * @param signature         | 签名名称,使用国内短信通用模板时填写
     * @return
     */
    static String buildRequestBody(String sender, String receiver, String templateId, String templateParas,
                                   String statusCallbackUrl, String signature) {
        if (null == sender || null == receiver || null == templateId || sender.isEmpty() || receiver.isEmpty()
                || templateId.isEmpty()) {
            System.out.println("buildRequestBody(): sender, receiver or templateId is null.");
            return null;
        }
        List<NameValuePair> keyValues = new ArrayList<NameValuePair>();

        keyValues.add(new BasicNameValuePair("from", sender));
        keyValues.add(new BasicNameValuePair("to", receiver));
        keyValues.add(new BasicNameValuePair("templateId", templateId));
        if (null != templateParas && !templateParas.isEmpty()) {
            keyValues.add(new BasicNameValuePair("templateParas", templateParas));
        }
        if (null != statusCallbackUrl && !statusCallbackUrl.isEmpty()) {
            keyValues.add(new BasicNameValuePair("statusCallback", statusCallbackUrl));
        }
        if (null != signature && !signature.isEmpty()) {
            keyValues.add(new BasicNameValuePair("signature", signature));
        }

        return URLEncodedUtils.format(keyValues, Charset.forName("UTF-8"));
    }

    /**
     * 构造X-WSSE参数值
     *
     * @param appKey
     * @param appSecret
     * @return
     */
    static String buildWsseHeader(String appKey, String appSecret) {
        if (null == appKey || null == appSecret || appKey.isEmpty() || appSecret.isEmpty()) {
            System.out.println("buildWsseHeader(): appKey or appSecret is null.");
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        //Created
        String time = sdf.format(new Date());
        //Nonce
        String nonce = UUID.randomUUID().toString().replace("-", "");
        byte[] passwordDigest = DigestUtils.sha256(nonce + time + appSecret);
        String hexDigest = Hex.encodeHexString(passwordDigest);
        //如果JDK版本是1.8,请加载原生Base64类,并使用如下代码
        //PasswordDigest
        String passwordDigestBase64Str = Base64.getEncoder().encodeToString(hexDigest.getBytes());
        return String.format(WSSE_HEADER_FORMAT, appKey, passwordDigestBase64Str, nonce, time);
    }
}



