package com.yeejoin.equipmanage.listener;

import com.alibaba.fastjson.JSONObject;
import com.yeejoin.amos.boot.biz.common.utils.RedisUtils;
import com.yeejoin.equipmanage.common.entity.EquipmentCategory;
import com.yeejoin.equipmanage.common.vo.IndustryVo;
import com.yeejoin.equipmanage.event.EquipmentTreeListEvent;
import com.yeejoin.equipmanage.service.IEquipmentCategoryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Set;

//import com.yeejoin.equipmanage.controller.EquipmentCategoryController;

@Component
//@EnableAsync
public class EquipmentTreeListListener implements ApplicationListener<EquipmentTreeListEvent> {

    private static final String equipmentTreeListAll = "equipmentTreeListAll";
    private static final String equipmentTreeListIndustry = "equipmentTreeListIndustry";

    @Autowired
    private RedisUtils redisUtils;

    @Autowired
    IEquipmentCategoryService iEquipmentCategoryService;

    @Async("equipAsyncExecutor")
    @Override
    public void onApplicationEvent(EquipmentTreeListEvent event) {
        // 更新redis全量装备树
        this.updateTreeListAll();
        // 更新redis行业过滤树形结构
        this.updateTreeListIndustry(event.idSet());
        this.updateTreeListIndustry(event.equipmentCategory());
        this.updateTreeListIndustryByType(event.typeSet());
    }

    /**
     * 更新全量树
     */
    private void updateTreeListAll() {
        boolean treeListAllFlag = redisUtils.hasKey(equipmentTreeListAll);
        if (treeListAllFlag) {
            redisUtils.del(equipmentTreeListAll);
        }
        List<IndustryVo> listAll = iEquipmentCategoryService.getTreeListAll();
        redisUtils.set(equipmentTreeListAll, JSONObject.toJSONString(listAll),0);
    }

    /**
     * 删除类别更新行业过滤缓存
     * @param idSet
     */
    private void updateTreeListIndustry(Set<String> idSet){
        if (idSet != null) {
            // 根据行业code缓存
            if (idSet.size()>0) {
                for (String code : idSet) {
                    updateTreeListIndustryByCode(code);
                }
            }
        }
    }

    /**
     * 删除类别更新行业过滤缓存
     * @param typeSet
     */
    private void updateTreeListIndustryByType(Set<String> typeSet){
        if (typeSet != null) {
            // 根据type
            if (typeSet.size()>0) {
                for (String type : typeSet) {
                    updateTreeListIndustryByType(type);
                }
            }
        }
    }

    /**
     * 新增修改缓存更新行业过滤
     * @param equipmentCategory
     */
    private void updateTreeListIndustry(EquipmentCategory equipmentCategory){
        if (equipmentCategory != null) {
            String code = equipmentCategory.getIndustryCode();
            if(code!=null && !"".equals(code)){
                updateTreeListIndustryByCode(code);
            }
            String codeType = equipmentCategory.getCode();
            if(codeType!=null && !"".equals(codeType)){
                updateTreeListIndustryByType(codeType.substring(0,1));
            }
        }
    }

    /**
     * 根据code更新缓存行业过滤
     */
    private void updateTreeListIndustryByCode(String code){
        String equipmentTreeListIndustryKey = equipmentTreeListIndustry + "-" + code;
        boolean treeListAllFlag = redisUtils.hasKey(equipmentTreeListIndustryKey);
        if (treeListAllFlag) {
            redisUtils.del(equipmentTreeListIndustryKey);
        }
        List<EquipmentCategory> listByCode = iEquipmentCategoryService.getIndustryTreeList(code);
        redisUtils.set(equipmentTreeListIndustryKey, JSONObject.toJSONString(listByCode),0);
    }

    /**
     * 根据不同装备
     */
    private void updateTreeListIndustryByType(String type){
        if ("2".equals(type)){
            String equipmentTreeListByType = "equipmentTreeListCL";
            boolean treeListAllFlag = redisUtils.hasKey(equipmentTreeListByType);
            if (treeListAllFlag) {
                redisUtils.del(equipmentTreeListByType);
            }
            List<EquipmentCategory>  list = iEquipmentCategoryService.getEquipmentCategoryCarList();
            redisUtils.set(equipmentTreeListByType, JSONObject.toJSONString(list),0);
        } else {
            String equipmentTreeListByType = "equipmentTreeListZB";
            boolean treeListAllFlag = redisUtils.hasKey(equipmentTreeListByType);
            if (treeListAllFlag) {
                redisUtils.del(equipmentTreeListByType);
            }
            List<EquipmentCategory>  list = iEquipmentCategoryService.getEquipmentCategoryEquipmentList();
            redisUtils.set(equipmentTreeListByType, JSONObject.toJSONString(list),0);
        }


    }
}
