/// 计算公式 - 管网供水能力
///
///
import 'package:flutter/material.dart';
import 'package:amos_flutter_ui/amos_flutter_ui.dart';

import './common_formula.dart';

/// 管网供水能力
class PipeWater extends StatefulWidget {
  const PipeWater({Key key}) : super(key: key);

  @override
  _PipeWaterState createState() => _PipeWaterState();
}

class _PipeWaterState extends State<PipeWater> {
  GlobalKey<FormState> oneformKeyformKey = GlobalKey<FormState>();
  Map<String, dynamic> _oneConfig = {'speed': 1.0};

  @override
  void initState() {
    super.initState();
  }

  @override
  Widget build(BuildContext context) {
    return FormulaButtonSubmit(
      title: '管网供水能力',
      formChild: AmosForm(
        formKey: oneformKeyformKey,
        children: renderTabOneForm(context),
        onChanged: (value) {
          _oneConfig = value;
        },
      ),
      onSubmit: onSubmit,
      onReset: onReset,
    );
  }

  List<Widget> renderTabOneForm(BuildContext context) {
    return <Widget>[
      FormItem(
        layout: FormLayout.vertical,
        label: '供水管网直径(m)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'diameter',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '管网内水流速',
        invert: true,
        formfield: AmosRadioGroup(
          fieldKey: 'speed',
          initialValue: _oneConfig['speed'],
          options: const [
            {'title': '支状管网1m/s', 'value': 1.0},
            {'title': '环状管网1.5m/s', 'value': 1.5}
          ],
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '消防车泵流量(L/s)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'count',
          maxLines: 1,
        ),
      )
    ];
  }


  void onSubmit(context) {
    if (!PipeValue.check(_oneConfig)) {
      shortMessage('计算参数不能为空');
      return;
    }
    PipeValue ffv = PipeValue.fromMap(_oneConfig);
    showBottomSheetDialog(
        context: context,
        builder: (BuildContext context) {
          return CalcResultPanel(results: PipeValue.getResult(ffv));
        });
  }

  void onReset() {
    oneformKeyformKey.currentState.reset();
  }
}

/// 管网供水能力
class PipeValue {
  /// 供水管网直径
  double diameter;

  /// 管网内水流速
  double speed;

  /// 消防车泵流量
  double count;

  PipeValue({this.diameter, this.speed, this.count});

  Map toJson() {
    Map map = <String, dynamic>{};
    map['speed'] = speed;
    map['count'] = count;
    map['diameter'] = diameter;
    return map;
  }

  static bool check(Map<String, dynamic> map) {
    String diameter = map['diameter'];
    double speed = map['speed'];
    String count = map['count'];

    if (diameter == null ||
        diameter.isEmpty ||
        count == null ||
        count.isEmpty ||
        speed == null) {
      return false;
    }
    return true;
  }

  static PipeValue fromMap(Map<String, dynamic> map) {
    PipeValue ffv = PipeValue(
        count: double.parse(map['count'] ?? ''),
        diameter: double.parse(map['diameter'] ?? ''),
        speed: map['speed']);
    return ffv;
  }

  /// 计算管网供水流量(L/s)
  static double calcOne(PipeValue ffv) {
    double liquid = double.parse(
        (ffv.diameter * ffv.diameter * 0.0008 * ffv.speed).toStringAsFixed(2));
    return liquid;
  }

  /// 计算停靠消防车数量(辆)
  static int calcTwo(PipeValue ffv) {
    double liquid = double.parse(
        (ffv.diameter * ffv.diameter * 0.0008 * ffv.speed).toStringAsFixed(2));
    return (liquid / ffv.count).ceil();
  }

  /// 计算结果
  static List<Map<String, dynamic>> getResult(PipeValue ffv) {
    double one = calcOne(ffv);
    int two = calcTwo(ffv);
    List<Map<String, dynamic>> res = [];
    res.add({'title': '管网供水流量', 'value': '$one L/s'});
    res.add({'title': '停靠消防车数量', 'value': '$two 辆'});
    return res;
  }
}
