import 'package:flutter/material.dart';
import 'dart:async';
import 'package:intl/intl.dart'; // 需添加intl依赖
import '../../services/qis.dart';

Map<String, dynamic> equipTimeEnums = {
  '超期': { 'bgColor': Color.fromRGBO(255, 238, 235, 1), 'textColor': Color.fromRGBO(255, 115, 87, 1) },
  '临期': { 'bgColor': Color.fromRGBO(255, 244, 224, 1), 'textColor': Color.fromRGBO(255, 162, 0, 1) },
  '正常': { 'bgColor': Color.fromRGBO(225, 247, 234, 1), 'textColor': Color.fromRGBO(1, 184, 82, 1) },
};

Map<String, String> eventEnums = {
  '未标记': '未标记',
  'CATEGORY_ADJUSTMENT': '管理类别调整',
  'VALIDITY_ADJUSTMENT': '有效期调整',
  'PERIODIC_ADJUSTMENT': '检定周期调整',
  'ENABLE': '设备启用',
  'ENABLE_SCRAP': '设备启用(报废确认)',
  'ENABLE_MAINTAIN': '设备启用(维修)',
  'ABNORMAL_SCRAP': '设备非正常报废',
  'ABNORMAL_SCRAP_SCRAP': '非正常报废(报废确认)',
  'ABNORMAL_SCRAP_MAINTAIN': '非正常报废(维修)',
  'SEALED': '设备封存',
  'DEFERRED_EXAMINATION': '设备缓检',
  'TRANSFER': '设备转移',
  'BEFORE_INSPECTION': '设备用前检定',
  'SELF': '自检计划',
  'OUTSIDE': '外送计划',
  'SPOTCHECK': '抽检计划'
};

Map<String, dynamic> equipStutasEnums = {

  '在用': { 'bgColor': Color.fromRGBO(225, 247, 234, 1), 'textColor': Color.fromRGBO(1, 184, 82, 1) },
  '缓检': { 'bgColor': Color.fromRGBO(225, 247, 234, 1), 'textColor': Color.fromRGBO(1, 184, 82, 1) },
  '封存': { 'bgColor': Color.fromRGBO(225, 247, 234, 1), 'textColor': Color.fromRGBO(1, 184, 82, 1) },
  '报废': { 'bgColor': Color.fromRGBO(156, 154, 154, 1), 'textColor': Color.fromRGBO(203, 200, 200, 1) },
};

List<Map<String, dynamic>> deviceList = [
  {'name': '计量编号', 'value': 'equipmentCode' },
  {'name': '设备名称', 'value': 'equipmentName' },
  {'name': '设备用途', 'value': 'equipmentUsage' },
  {'name': '生产厂家', 'value': 'manufacturerDate' },
  {'name': '出厂日期', 'value': 'manageCategory' },
  {'name': '管理类别', 'value': 'materialNumber' },
  {'name': '物料号', 'value': 'materialName' },
  {'name': '物料名称', 'value': 'equipmentCode' },
  {'name': '使用部门', 'value': 'userDepartment' },
  {'name': '出厂编号', 'value': 'manufacturerNumber' },
  {'name': '学科大类', 'value': 'majorSubjects' },
  {'name': '学科小类', 'value': 'subjectCategory' },
  {'name': '规格型号', 'value': 'specificationModel' },
  {'name': '测量范围', 'value': 'measureRange' },
  {'name': '分度值', 'value': 'divisionValue' },
  {'name': '准确度等级', 'value': 'accuracyLevel' },
  {'name': '允许误差', 'value': 'allowableError' },
  {'name': '准确度', 'value': 'accuracy' },
  {'name': '专用项目', 'value': 'earmarkedProject' },
  {'name': '确认间隔', 'value': 'confirmInterval' },
  {'name': '检定周期(月)', 'value': 'verifyCycle' },
  {'name': '检定方式', 'value': 'verifyMethod' },
  {'name': '是否强检', 'value': 'mandatoryInspect' },
  {'name': '是否标准器', 'value': 'etalon' },
  {'name': '设备状态', 'value': 'equipmentStatus' },
  {'name': '检定日期', 'value': 'lastVerifyDate' },
  {'name': '有效期', 'value': 'validityPeriod' },
  {'name': '检定人', 'value': 'inspector' },
  // {'name': '超期临期状态', 'value': 'overdueOrApproachingDue' },
  {'name': '参与业务', 'value': 'event' }

];

// 设备详情数据模型（适配低版本，不用空安全）
class DeviceInfo {
  String equipmentCode; // 计量编号
  String equipmentName; // 设备名称
  String equipmentUsage; // 设备用途
  String manufacturer; // 生产厂家
  String manufacturerDate; // 出厂日期
  String manageCategory; // 管理类别
  String materialNumber; // 物料号
  String materialName; // 物料名称
  String userDepartment; // 使用部门
  String manufacturerNumber; // 出厂编号
  String majorSubjects; // 学科大类
  String subjectCategory; // 学科小类
  String specificationModel; // 规格型号
  String measureRange; // 测量范围
  String divisionValue; // 分度值
  String accuracyLevel; // 准确度等级
  String allowableError; // 允许误差
  String accuracy; // 准确度
  String earmarkedProject; // 专用项目
  String confirmInterval; // 确认间隔
  String verifyCycle; // 检定周期
  String verifyMethod; // 检定方式
  String mandatoryInspect; // 是否强检
  String etalon; // 是否标准器
  String equipmentStatus; // 设备状态
  String lastVerifyDate; // 检定日期
  String validityPeriod; // 有效期
  String inspector; // 检定人
  String overdueOrApproachingDue; // 超期临期状态
  String event; // 参与业务

  // 构造方法（低版本用@required）
  DeviceInfo({
    this.equipmentCode,// 计量编号
    this.equipmentName,// 设备名称
    this.equipmentUsage,// 设备用途
    this.manufacturer,// 生产厂家
    this.manufacturerDate,// 出厂日期
    this.manageCategory,// 管理类别
    this.materialNumber,// 物料号
    this.materialName,// 物料名称
    this.userDepartment,// 使用部门
    this.manufacturerNumber,// 出厂编号
    this.majorSubjects,// 学科大类
    this.subjectCategory,// 学科小类
    this.specificationModel,// 规格型号
    this.measureRange,// 测量范围
    this.divisionValue,// 分度值
    this.accuracyLevel,// 准确度等级
    this.allowableError,// 允许误差
    this.accuracy,// 准确度
    this.earmarkedProject,// 专用项目
    this.confirmInterval,// 确认间隔
    this.verifyCycle,// 检定周期
    this.verifyMethod,// 检定方式
    this.mandatoryInspect,// 是否强检
    this.etalon,// 是否标准器
    this.equipmentStatus,// 设备状态
    this.lastVerifyDate,// 检定日期
    this.validityPeriod,// 有效期
    this.inspector,// 检定人
    this.overdueOrApproachingDue,// 超期临期状态
    this.event// 参与业务
  });
}

// 设备详情页面（核心）
class DeviceDetailPage extends StatefulWidget {
  final String deviceCode; // 接收扫码传递的设备编码

  // 构造方法：必传设备编码
  DeviceDetailPage({Key key, @required this.deviceCode}) : super(key: key);

  @override
  _DeviceDetailPageState createState() => _DeviceDetailPageState();
}

class _DeviceDetailPageState extends State<DeviceDetailPage> {
  bool _isLoading = true; // 加载状态
  Map<String, dynamic> _deviceInfo; // 设备详情数据
  String _errorMsg = ""; // 错误提示

  @override
  void initState() {
    super.initState();
    // 初始化时获取设备详情
    _fetchDeviceDetail();
  }

  // 模拟接口请求获取设备详情（替换为真实接口即可）
  Future<void> _fetchDeviceDetail() async {
    try {
      // 模拟网络请求延迟
      // await Future.delayed(Duration(seconds: 1));
      Map<String, dynamic> data = await API.queryEquipDetailsByCode(widget.deviceCode);
      if (data.isEmpty) {
        setState(() {
          _isLoading = false;
          _errorMsg = "获取设备详情失败";
        });
        return;
      }
      // 模拟根据设备编码返回数据（真实场景替换为http请求）
      String manufacturerDate = data['manufacturerDate'].isNotEmpty == true ? (int.tryParse(data['manufacturerDate']) != null ? DateFormat('yyyy-MM-dd').format(DateTime.fromMillisecondsSinceEpoch(int.parse(data['manufacturerDate']).toString().length == 10 ? int.parse(data['manufacturerDate'])*1000 : int.parse(data['manufacturerDate']))) : '') : '';
      String lastVerifyDate = data['lastVerifyDate'].isNotEmpty == true ? (int.tryParse(data['lastVerifyDate']) != null ? DateFormat('yyyy-MM-dd').format(DateTime.fromMillisecondsSinceEpoch(int.parse(data['lastVerifyDate']).toString().length == 10 ? int.parse(data['lastVerifyDate'])*1000 : int.parse(data['lastVerifyDate']))) : '') : '';
      String validityPeriod = data['validityPeriod'].isNotEmpty == true ? (int.tryParse(data['validityPeriod']) != null ? DateFormat('yyyy-MM-dd').format(DateTime.fromMillisecondsSinceEpoch(int.parse(data['validityPeriod']).toString().length == 10 ? int.parse(data['validityPeriod'])*1000 : int.parse(data['validityPeriod']))) : '') : '';
      String eventType = data['ledgerEvent'] != null ? data['ledgerEvent']['eventType'] ?? '未标记' : '未标记';
      Map<String, dynamic> info = {
        'equipmentCode': widget.deviceCode ?? '',
        // 计量编号
        'equipmentName': data['equipmentName'] ?? '',
        // 设备名称
        'equipmentUsage': (data['equipmentUsageLabel'] as List).map((
            e) => e['name']).toList().join(',') ?? '',
        // 设备用途
        'manufacturer': data['manufacturer'] ?? '',
        // 生产厂家
        'manufacturerDate': manufacturerDate ?? '', // 出厂日期
        'manageCategory': data['manageCategoryLabel'] != null ? data['manageCategoryLabel']['name'] ?? '' : '',
        // 管理类别
        'materialNumber': data['materialNumber'] ?? '',
        // 物料号
        'materialName': data['materialName'] ?? '',
        // 物料名称
        'userDepartment': (data['userDepartmentLabel'] as List).map((
            e) => e['title']).toList().join(',') ?? '',
        // 使用部门
        'manufacturerNumber': data['manufacturerNumber'] ?? '',
        // 出厂编号
        'majorSubjects': data['majorSubjectsLabel'] != null
            ? data['majorSubjectsLabel']['name'] ?? ''
            : '',
        // 学科大类
        'subjectCategory': data['subjectCategoryLabel'] != null
            ? data['subjectCategoryLabel']['name'] ?? ''
            : '',
        // 学科小类
        'specificationModel': data['specificationModel'] ?? '',
        // 规格型号
        'measureRange': data['measureRange'] ?? '',
        // 测量范围
        'divisionValue': data['divisionValue'] ?? '',
        // 分度值
        'accuracyLevel': data['accuracyLevel'] ?? '',
        // 准确度等级
        'allowableError': data['allowableError'] ?? '',
        // 允许误差
        'accuracy': data['accuracy'] ?? '',
        // 准确度
        'earmarkedProject': data['earmarkedProjectLabel'] != null
            ? data['earmarkedProjectLabel']['name'] ?? ''
            : '',
        // 专用项目
        'confirmInterval': data['confirmIntervalLabel'] != null
            ? data['confirmIntervalLabel']['name'] ?? ''
            : '',
        // 确认间隔
        'verifyCycle': (data['verifyCycle'] ?? '0').toString(),
        // 检定周期
        'verifyMethod': data['verifyMethodLabel'] != null
            ? data['verifyMethodLabel']['name'] ?? ''
            : '',
        // 检定方式
        'mandatoryInspect': data['mandatoryInspectLabel'] != null
            ? data['mandatoryInspectLabel']['name'] ?? ''
            : '',
        // 是否强检
        'etalon': data['etalonLabel'] != null
            ? data['etalonLabel']['name'] ?? ''
            : '',
        // 是否标准器
        'equipmentStatus': data['equipmentStatus'],
        // 设备状态
        'lastVerifyDate': lastVerifyDate ?? '', // 检定日期
        'validityPeriod': validityPeriod ?? '', // 有效期
        'inspector': data['inspector'] ?? '',
        // 检定人
        'validityPeriodStatus': data['validityPeriodStatus'] ?? '临期',
        // 超期临期状态
        'event': eventEnums[eventType] ?? '未标记'
        // 参与业务
      };

      setState(() {
        _isLoading = false;
        _deviceInfo = info;
      });
    } catch (e) {
      setState(() {
        _isLoading = false;
        _errorMsg = "获取设备详情失败：${e.toString()}";
      });
    }
  }

  // 构建加载中视图
  Widget _buildLoadingView() {
    return Center(
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          CircularProgressIndicator(),
          SizedBox(height: 16),
          Text("正在加载设备详情..."),
        ],
      ),
    );
  }

  // 构建错误视图
  Widget _buildErrorView() {
    return Center(
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          Icon(Icons.error_outline, color: Colors.red, size: 48),
          SizedBox(height: 16),
          Text(
            _errorMsg,
            textAlign: TextAlign.center,
            style: TextStyle(color: Colors.red),
          ),
          SizedBox(height: 16),
          ElevatedButton(
            onPressed: () {
              setState(() {
                _isLoading = true;
                _errorMsg = "";
              });
              _fetchDeviceDetail(); // 重新加载
            },
            child: Text("重新加载"),
          ),
        ],
      ),
    );
  }
// 基础Tag组件（可直接复用）
  Widget buildBasicTag(String text, dynamic bgColor, dynamic textColor) {
    Color _bgColor = bgColor != null ? bgColor as Color : Colors.black87;
    Color _textColor = textColor != null ? textColor as Color : Colors.black87;
    return Container(
      padding: EdgeInsets.symmetric(horizontal: 8, vertical: 4),
      margin: EdgeInsets.only(right: 6),
      decoration: BoxDecoration(
        color: _bgColor, // 浅绿背景
        borderRadius: BorderRadius.circular(4), // 圆角
      ),
      child: Text(
        text,
        style: TextStyle(
          fontSize: 12,
          color: _textColor, // 深绿文字
        ),
      ),
    );
  }

  // 构建设备详情视图
  Widget _buildDetailView() {
    Map validityPeriodStatusItem = equipTimeEnums[_deviceInfo['validityPeriodStatus']];
    Map equipmentStatusItem = equipStutasEnums[_deviceInfo['equipmentStatus']];
    return ListView(
      padding: EdgeInsets.all(12),
      children: [
        // 设备编码卡片（突出显示）
        Card(
          elevation: 2,
          shape: RoundedRectangleBorder(
            borderRadius: BorderRadius.circular(8), // 圆角卡片
          ),
          child: Padding(
            padding: EdgeInsets.all(16),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  _deviceInfo['equipmentName'],
                  style: TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
                ),
                SizedBox(height: 8),
                Row(
                    // alignment: WrapAlignment.center,
                   children: [
                    Text(
                      _deviceInfo['equipmentCode'],
                      style: TextStyle(color: Colors.grey, fontSize: 14),
                    ),
                     SizedBox(width: 8),
                     buildBasicTag(
                         _deviceInfo['equipmentStatus'],
                         equipmentStatusItem.isNotEmpty ? equipmentStatusItem['bgColor'] : null,
                         equipmentStatusItem.isNotEmpty ? equipmentStatusItem['textColor'] : null
                     ),
                    if(_deviceInfo['validityPeriodStatus'] != '正常') buildBasicTag(
                        _deviceInfo['validityPeriodStatus'],
                        validityPeriodStatusItem.isNotEmpty ? validityPeriodStatusItem['bgColor'] : null,
                        validityPeriodStatusItem.isNotEmpty ? validityPeriodStatusItem['textColor'] : null
                    )
                  ]
                )

              ],
            ),
          ),
        ),
        SizedBox(height: 8),

        // 设备基本信息列表
        Card( // 主卡片容器，匹配页面样式
          elevation: 2, // 轻微阴影，贴近原生样式
          shape: RoundedRectangleBorder(
            borderRadius: BorderRadius.circular(8), // 圆角卡片
          ),
          child: Padding(
            padding: EdgeInsets.symmetric(horizontal: 1, vertical: 12),
            child: Column(
              children: deviceList.map((entry) {
                // 遍历数据，生成键值对列表项
                return Padding(
                  padding: EdgeInsets.symmetric(vertical: 8),
                  child: Row(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      // 左侧标签（固定宽度，右对齐）
                      SizedBox(
                        width: 90,
                        child: Text(
                          "${entry['name']}：",
                          style: TextStyle(
                            fontSize: 15,
                            color: Colors.black87, // 标签灰色
                          ),
                          textAlign: TextAlign.right,
                        ),
                      ),
                      SizedBox(width: 12), // 间距
                      // 右侧值（自适应宽度，左对齐）
                      Expanded(
                        child: Text(
                          _deviceInfo[entry['value']] != null ?_deviceInfo[entry['value']] : '',
                          style: TextStyle(
                            fontSize: 15,
                            color: Colors.black54, // 值黑色
                          ),
                        ),
                      ),
                    ],
                  ),
                );
              }).toList(),
            ),
          ),
        ),
      ],
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Text("设备详情"),
        // 低版本AppBar返回按钮适配
        leading: IconButton(
          icon: Icon(Icons.arrow_back),
          onPressed: () => Navigator.pop(context),
        ),
      ),
      body: RefreshIndicator(
        onRefresh: _fetchDeviceDetail,
        child:  _isLoading
            ? _buildLoadingView() // 加载中
            : _errorMsg.isNotEmpty
            ? _buildErrorView() // 加载失败
            : _buildDetailView(), // 加载成功
      )

    );
  }
}