/// 计算公式 - 液化烃球罐冷却用水量
///
///
import 'package:flutter/material.dart';
import 'package:amos_flutter_ui/amos_flutter_ui.dart';
import './common_formula.dart';

import './vertical_tank_water.dart';

/// 液化烃球罐冷却用水量
class HydrocarbonTank extends StatefulWidget {
  const HydrocarbonTank({Key key}) : super(key: key);

  @override
  _HydrocarbonTankState createState() => _HydrocarbonTankState();
}

class _HydrocarbonTankState extends State<HydrocarbonTank>
    with SingleTickerProviderStateMixin {
  TabController tabController;
  GlobalKey<FormState> oneformKeyformKey = GlobalKey<FormState>();
  Map<String, dynamic> _oneConfig = {'strength': '0.2', 'time': '6'};

  GlobalKey<FormState> twoformKeyformKey = GlobalKey<FormState>();
  Map<String, dynamic> _twoConfig = {'strength': '0.2', 'time': '6'};

  GlobalKey<FormState> threeformKeyformKey = GlobalKey<FormState>();
  Map<String, dynamic> _threeConfig = {};

  @override
  void initState() {
    super.initState();
    tabController = TabController(length: 3, vsync: this);
  }

  @override
  Widget build(BuildContext context) {
    return FormulaButtonSubmit(
        title: '液化烃球罐冷却用水量',
        onSubmit: onSubmit,
        onReset: onReset,
        body: Container(
            child: Column(
          children: [
            Container(
                height: 42,
                child: TabBar(
                  labelColor: Colors.black,
                  isScrollable: true,
                  unselectedLabelColor: Colors.black54,
                  indicatorSize: TabBarIndicatorSize.label,
                  indicatorColor: Colors.lightBlue,
                  tabs: const [
                    Tab(text: '着火罐冷却用水量'),
                    Tab(text: '冷近罐冷却用水量'),
                    Tab(text: '冷却枪炮数量')
                  ],
                  controller: tabController,
                )),
            Expanded(
                child: TabBarView(
              children: [
                CommonFormulaContent(
                  formChild: AmosForm(
                    formKey: oneformKeyformKey,
                    children: renderTabOneForm(context),
                    onChanged: (value) {
                      _oneConfig = value;
                    },
                  ),
                  hasFooter: false,
                ),
                CommonFormulaContent(
                  formChild: AmosForm(
                    formKey: twoformKeyformKey,
                    children: renderTabTwoForm(context),
                    onChanged: (value) {
                      _twoConfig = value;
                    },
                  ),
                  hasFooter: false,
                ),
                CommonFormulaContent(
                  formChild: AmosForm(
                    formKey: threeformKeyformKey,
                    children: renderTabThreeForm(context),
                    onChanged: (value) {
                      _threeConfig = value;
                    },
                  ),
                  hasFooter: false,
                ),
              ],
              controller: tabController,
            ))
          ],
        )));
  }

  List<Widget> renderTabOneForm(BuildContext context) {
    return <Widget>[
      FormItem(
        layout: FormLayout.vertical,
        label: '着火罐直径(m)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'diameter',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '着火罐冷却水供给强度(L/s*m)',
        invert: true,
        formfield: AmosTextNumberField(
          initialValue: _oneConfig['strength'],
          fieldKey: 'strength',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '着火罐数量(个)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'count',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '冷却持续时间(小时)',
        invert: true,
        formfield: AmosTextNumberField(
          initialValue: _oneConfig['time'],
          fieldKey: 'time',
          maxLines: 1,
        ),
      )
    ];
  }

  List<Widget> renderTabTwoForm(BuildContext context) {
    return <Widget>[
      FormItem(
        layout: FormLayout.vertical,
        label: '冷近罐直径(m)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'diameter',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '冷近罐冷却水供给强度(L/s·m²)',
        invert: true,
        formfield: AmosTextNumberField(
          initialValue: _twoConfig['strength'],
          fieldKey: 'strength',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '冷近罐数量(个)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'count',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '冷却持续时间(小时)',
        invert: true,
        formfield: AmosTextNumberField(
          initialValue: _twoConfig['time'],
          fieldKey: 'time',
          maxLines: 1,
        ),
      )
    ];
  }

  List<Widget> renderTabThreeForm(BuildContext context) {
    return <Widget>[
      FormItem(
        layout: FormLayout.vertical,
        label: '水枪流量(m)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'gunsCount',
          maxLines: 1,
        ),
      ),
      FormItem(
        layout: FormLayout.vertical,
        label: '移动炮流量(m)',
        invert: true,
        formfield: AmosTextNumberField(
          fieldKey: 'moveGunsCount',
          maxLines: 1,
        ),
      )
    ];
  }

  void onSubmit(context) {
    if (tabController.index == 0) {
      if (!CatchFireValue.check(_oneConfig)) {
        shortMessage('计算参数不能为空');
        return;
      }
      CatchFireValue ffv = CatchFireValue.fromMap(_oneConfig);
      showBottomSheetDialog(
          context: context,
          builder: (BuildContext context) {
            return CalcResultPanel(results: CatchFireValue.getResult(ffv));
          });
    }
    if (tabController.index == 1) {
      if (!ColdFireValue.check(_twoConfig)) {
        shortMessage('计算参数不能为空');
        return;
      }
      ColdFireValue ffv = ColdFireValue.fromMap(_twoConfig);
      showBottomSheetDialog(
          context: context,
          builder: (BuildContext context) {
            return CalcResultPanel(results: ColdFireValue.getResult(ffv));
          });
    }
    if (tabController.index == 2) {
      if (!ColdgunsCountValue.check(_threeConfig)) {
        shortMessage('计算参数不能为空');
        return;
      }
      ColdgunsCountValue ffv = ColdgunsCountValue.fromMap(_threeConfig);
      showBottomSheetDialog(
          context: context,
          builder: (BuildContext context) {
            return CalcResultPanel(results: ColdgunsCountValue.getResult(ffv));
          });
    }
  }

  void onReset() {
    if (tabController.index == 0) {
      oneformKeyformKey.currentState.reset();
    }
    if (tabController.index == 1) {
      twoformKeyformKey.currentState.reset();
    }
    if (tabController.index == 2) {
      threeformKeyformKey.currentState.reset();
    }
  }
}

/// 冷却枪炮数量
class ColdgunsCountValue {
  /// 水枪流量
  double gunsCount;

  /// 移动炮流量
  double moveGunsCount;

  ColdgunsCountValue({this.gunsCount, this.moveGunsCount});

  Map toJson() {
    Map map = <String, dynamic>{};
    map['gunsCount'] = gunsCount;
    map['moveGunsCount'] = moveGunsCount;
    return map;
  }

  static bool check(Map<String, dynamic> map) {
    String gunsCount = map['gunsCount'];
    String moveGunsCount = map['moveGunsCount'];

    if (gunsCount == null ||
        gunsCount.isEmpty ||
        moveGunsCount == null ||
        moveGunsCount.isEmpty) {
      return false;
    }
    return true;
  }

  static ColdgunsCountValue fromMap(Map<String, dynamic> map) {
    ColdgunsCountValue ffv = ColdgunsCountValue(
        gunsCount: double.parse(map['gunsCount'] ?? ''),
        moveGunsCount: double.parse(map['moveGunsCount'] ?? ''));
    return ffv;
  }

  /// 计算冷却水枪数量(支)
  static int calcOne(ColdgunsCountValue ffv) {
    int liquid = (1000 / ffv.gunsCount).ceil();
    return liquid;
  }

  /// 计算冷却移动炮数量(门)
  static int calcTwo(ColdgunsCountValue ffv) {
    int liquid = (1000 / ffv.moveGunsCount).ceil();
    return liquid;
  }

  /// 计算结果
  static List<Map<String, dynamic>> getResult(ColdgunsCountValue ffv) {
    int one = calcOne(ffv);
    int two = calcTwo(ffv);
    List<Map<String, dynamic>> res = [];
    res.add({'title': '冷却水枪数量', 'value': '$one 支'});
    res.add({'title': '冷却移动炮数量', 'value': '$two 门'});
    return res;
  }
}
