package com.yeejoin.amos.fas.business.service.impl;


import static org.hamcrest.CoreMatchers.either;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONObject;
import com.itextpdf.text.pdf.PdfStructTreeController.returnType;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;
import com.google.common.collect.Lists;
import com.yeejoin.amos.fas.business.bo.BindPointBo;
import com.yeejoin.amos.fas.business.bo.CheckErrorBo;
import com.yeejoin.amos.fas.business.bo.RiskPointRpnChangeBo;
import com.yeejoin.amos.fas.business.dao.mapper.PatrolMapper;
import com.yeejoin.amos.fas.business.dao.repository.*;
import com.yeejoin.amos.fas.business.service.intfc.IView3dService;
import com.yeejoin.amos.fas.business.vo.ErrorContentVo;
import com.yeejoin.amos.fas.business.vo.SafetyIndexDetailVo;
import com.yeejoin.amos.fas.business.vo.TodaySafetyIndexVo;
import com.yeejoin.amos.fas.common.enums.CheckStatusEnum;
import com.yeejoin.amos.fas.common.enums.RpnChangeTypeEum;
import com.yeejoin.amos.fas.common.enums.StatisticsErrorTypeEum;
import com.yeejoin.amos.fas.core.common.request.DateUtil;
import com.yeejoin.amos.fas.core.util.*;
import com.yeejoin.amos.fas.dao.entity.*;
import com.yeejoin.amos.fas.exception.YeeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import com.yeejoin.amos.fas.business.dao.mapper.View3dMapper;
import com.yeejoin.amos.fas.common.enums.ResourceTypeDefEnum;
import com.yeejoin.amos.fas.core.common.response.RegionTreeResponse;
import com.yeejoin.amos.fas.core.util.CommonResponse;
import com.yeejoin.amos.fas.core.util.CommonResponseUtil;

/**
 * @author DELL
 */
@Service("view3dService")
public class View3dServiceImpl implements IView3dService {
	
	Logger log = LoggerFactory.getLogger(View3dServiceImpl.class);

	@Autowired
	private View3dMapper view3dMapper;
    @Autowired
    private IRiskSourceDao iRiskSourceDao;
    @Autowired
    private IFireCarDao iFireCarDao;
    @Autowired
    private IFireEquipmentDao iFireEquipmentDao;
    @Autowired
    private IFireStationDao iFireStationDao;
    @Autowired
    private IEquipmentDao iEquipmentDao;
    @Autowired
    private IWaterResourceDao iWaterResourceDao;
    @Autowired
    private PatrolMapper patrolMapper;
    @Autowired
    private IRiskLevelDao iRiskLevelDao;
    @Value("${param.system.online.date}")
    private String onLineDate;



    @Override
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse setPoint3dPosition(List<BindPointBo> pointBoList) {
        if (CollectionUtils.isEmpty(pointBoList)){
            return CommonResponseUtil.failure("点不能为空");
        }
        pointBoList.forEach(pointBo ->{
            this.savePointPositionByType(pointBo.getPointType(),pointBo);
        });
        return CommonResponseUtil.success();
    }

    public void savePointPositionByType(String pointType,BindPointBo pointBo){
        switch (pointType){
            case "riskSource":
                this.updateRiskSourcePosition(pointBo);
                break;
            case "patrol":
                this.updatePatrolPosition(pointBo);
                break;
            case "impEquipment":
                this.updateEquipmentPosition(pointBo);
                break;
            case "monitorEquipment":
            case "video":
            case "fireEquipment":
                this.updateFireEquipmentPosition(pointBo);
                break;
            case "fireCar":
                this.updateFireCarPosition(pointBo);
            case "fireChamber":
            case "fireFoamRoom":
                this.updateFireStationPosition(pointBo);
                break;
            case "hydrant":
            case "pool":
                this.updateWaterSourcePosition(pointBo);
                break;
            default:
                log.error("不支持的类型-->" + pointType);
                break;
        }
    }

    public void updateFireCarPosition(BindPointBo pointBo) {
       Optional<FireCar> data = iFireCarDao.findById(pointBo.getPointId());
       if(data.isPresent()){
           FireCar fireCar = data.get();
           fireCar.setUe4Location(pointBo.getUe4Location());
           fireCar.setUe4Rotation(pointBo.getUe4Rotation());
           iFireCarDao.save(fireCar);
       }
    }

    public void updateWaterSourcePosition(BindPointBo pointBo) {
        Optional<WaterResource> data = iWaterResourceDao.findById(pointBo.getPointId());
        if(data.isPresent()){
            WaterResource waterResource = data.get();
            waterResource.setUe4Location(pointBo.getUe4Location());
            waterResource.setUe4Rotation(pointBo.getUe4Rotation());
            iWaterResourceDao.save(waterResource);
        }
    }

    public void updateFireStationPosition(BindPointBo pointBo) {
        Optional<FireStation> data = iFireStationDao.findById(pointBo.getPointId());
        if(data.isPresent()){
            FireStation fireStation = data.get();
            fireStation.setUe4Location(pointBo.getUe4Location());
            fireStation.setUe4Rotation(pointBo.getUe4Rotation());
            iFireStationDao.save(fireStation);
        }
    }

    public void updateFireEquipmentPosition(BindPointBo pointBo) {
       Optional<FireEquipment> data = iFireEquipmentDao.findById(pointBo.getPointId());
       if(data.isPresent()){
           FireEquipment fireEquipment = data.get();
           fireEquipment.setUe4Location(pointBo.getUe4Location());
           fireEquipment.setUe4Rotation(pointBo.getUe4Rotation());
           iFireEquipmentDao.save(fireEquipment);
       }
    }

    public void updateEquipmentPosition(BindPointBo pointBo) {
        Optional<Equipment> data=iEquipmentDao.findById(pointBo.getPointId());
        Equipment  equipment = null;
        if(data.isPresent()){
            equipment = data.get();
            equipment.setUe4Location(pointBo.getUe4Location());
            equipment.setUe4Rotation(pointBo.getUe4Rotation());
            iEquipmentDao.save(equipment);
        }
    }

    public void updatePatrolPosition(BindPointBo pointBo) {
        patrolMapper.updatePoint3dPosition(pointBo);
    }

    public void updateRiskSourcePosition(BindPointBo pointBo) {
        Optional<RiskSource> data = iRiskSourceDao.findById(pointBo.getPointId());
        if(data.isPresent()){
            RiskSource riskSource = data.get();
            riskSource.setUe4Location(pointBo.getUe4Location());
            riskSource.setUe4Rotation(pointBo.getUe4Rotation());
            iRiskSourceDao.save(riskSource);
        }
    }

	@Override
	public List<Map<String, String>> getPointType() {
		return ResourceTypeDefEnum.getTypeList();
	}

	@Override
	public List<RegionTreeResponse> getPointTreeByType(String type,String orgCode) {
		List<RegionTreeResponse> pointTreeByType = view3dMapper.getPointTreeByType(type,orgCode);
		return convertPointTree(pointTreeByType);
	}
	
	//区域列表转树
	private static List<RegionTreeResponse> convertPointTree(List<RegionTreeResponse> list) {
		HashMap<Long, RegionTreeResponse> hashMap = new HashMap<>();
		list.forEach( e -> hashMap.put(e.getId(), e));
		List<RegionTreeResponse> treeList = new ArrayList<RegionTreeResponse>();
        list.forEach( r -> {
        	Long parentId = r.getParentId();
        	if(parentId != null) {
        		RegionTreeResponse pRegion = hashMap.get(parentId);
        		if(pRegion != null) {
        			pRegion.getChildren().add(r);
        		}else {
        			treeList.add(r);
        		}
        	}
        });
        return treeList;
    }

	@Override
	public HashMap<String, Object> getPointDetailByTypeAndId(String type, Long pointId,String orgCode) {
		return view3dMapper.getPointDetailByTypeAndId(type,pointId,orgCode);
	}

    @Override
    public TodaySafetyIndexVo getSafetyIndexInfoByDate(String orgCode,String date) {
        if(!StringUtil.isNotEmpty(date)){
            date = DateUtil.getDateFormat(new Date(),DateUtil.DATE_DEFAULT_FORMAT);
        }
        Optional<RiskSource> optional = iRiskSourceDao.findByOrgCodeAndParentId(orgCode,0L);
        TodaySafetyIndexVo vo = new TodaySafetyIndexVo();
        if(optional.isPresent()){
            //1.按照等级进行转换rpn为分数-机构
            RiskSource riskSource = optional.get();
            double safetyIndex = this.calSafetyIndex(riskSource);
            vo.setSafetyIndex(safetyIndex);
            //2.统计风险上升异常数量（风险点）-日期+机构
            Long upperNum = this.getRiskUpperNum(orgCode, date);
            vo.setRiskExceptionNum(upperNum);
            //3.统计巡检异常数量（不合格、漏检）-日期+机构
            Long checkExceptionNum = this.getCheckExceptionNum(orgCode,date);
            vo.setCheckExceptionNum(checkExceptionNum);
            //4.统计火灾报警数量-日期+机构
            //TODO：待毛颖确认调整完成后继续
            vo.setFireExceptionNum(0L);
        } else {
            throw new YeeException("机构："+orgCode+" 不存在");
        }
        return vo;
    }

    @Override
    public List<SafetyIndexDetailVo> getSafetyIndexDetail(String type,String orgCode) {
        List<SafetyIndexDetailVo> resultList = Lists.newArrayList();
        String dateStr = DateUtil.getDateFormat(new Date(),DateUtil.DATE_DEFAULT_FORMAT);
        Map<String,Object> params = new HashMap<>();
        params.put("orgCode",orgCode);
        params.put("date",dateStr);
        if(StatisticsErrorTypeEum.risk.getCode().equals(type)){
            //1.按照日期+机构，查询出所有风险点
            List<RiskPointRpnChangeBo> risks = view3dMapper.getAllErrorRiskPoint(params);
            //2.按照 风险等级id进行分组，加工数据
            Map<Long,List<RiskPointRpnChangeBo>> map = risks.stream().collect(Collectors.groupingBy(RiskPointRpnChangeBo::getRiskLevelId));
            List<RiskLevel> riskLevelList = iRiskLevelDao.findAll();
            Map<Long,String> riskLevelMap = riskLevelList.stream().collect(Collectors.toMap(RiskLevel::getId,RiskLevel::getName));
            map.forEach((k,v)->{
                SafetyIndexDetailVo safetyIndexDetailVo = new SafetyIndexDetailVo();
                safetyIndexDetailVo.setTypeCode(Long.toString(k));
                safetyIndexDetailVo.setTypeName(riskLevelMap.get(k));
                safetyIndexDetailVo.setContent(getErrorContentVos(v));
                resultList.add(safetyIndexDetailVo);
            });
        } else if(StatisticsErrorTypeEum.check.getCode().equals(type)){
            //1.按照日期+机构，查询出异常巡检
            List<CheckErrorBo> checkErrorBoList = view3dMapper.getAllErrorPatrolPoint(params);
            //2.按照状态进行分组，加工返回数据
            Map<String,List<CheckErrorBo>> map = checkErrorBoList.stream().collect(Collectors.groupingBy(CheckErrorBo::getStatus));
            map.forEach((k,v)->{
                SafetyIndexDetailVo safetyIndexDetailVo = new SafetyIndexDetailVo();
                safetyIndexDetailVo.setTypeCode(k);
                safetyIndexDetailVo.setTypeName(CheckStatusEnum.getEnum(k).getName());
                safetyIndexDetailVo.setContent(getErrorContentVos(v));
                resultList.add(safetyIndexDetailVo);
            });
        } else if(StatisticsErrorTypeEum.equip.getCode().equals(type)){
            //TODO：待毛颖确认调整完成后继续
        }
        return resultList;
    }

    private List<ErrorContentVo> getErrorContentVos(List<?> v) {
        List<ErrorContentVo> errorContentVoList = new ArrayList<>();
        errorContentVoList = v.stream().map(e->{
            ErrorContentVo errorContentVo = new ErrorContentVo();
            BeanUtils.copyProperties(e,errorContentVo);
            return errorContentVo;
        }).collect(Collectors.toList());
        return errorContentVoList;
    }

    private Long getCheckExceptionNum(String orgCode, String date) {
        Map<String,Object> params = new HashMap<>();
        params.put("orgCode", orgCode);
        params.put("date", date);
        return view3dMapper.countCheckException(params);
    }

    private Long getRiskUpperNum(String orgCode, String date) {
        Map<String,Object> param = new HashMap<>();
        param.put("type", RpnChangeTypeEum.upper.getValue());
        param.put("orgCode", orgCode);
        param.put("date", date);
        return view3dMapper.countUpperRiskPoint(param);
    }

    private double calSafetyIndex(RiskSource riskSource) {
        BigDecimal rpn = new BigDecimal(String.valueOf(riskSource.getRpn() == null ? "0" : riskSource.getRpn()));
        //计算规则：除10 减100 求绝对值,结果保留1位小数
        return rpn.divide(new BigDecimal("10")).subtract(new BigDecimal("100")).abs().setScale(1,BigDecimal.ROUND_HALF_UP).doubleValue();
    }

	@Override
	public HashMap<String, Object> getSafetyIndexWeek(String orgCode) {
		return view3dMapper.getSafetyIndexWeek(orgCode);
	}

    @Override
    public Map<String, Object> getSystemOnlineDate(String orgCode) {
        Date beginDate = DateUtil.formatDate(onLineDate,DateUtil.DATE_DEFAULT_FORMAT);
        Date now = DateUtil.getNow();
        return DateUtil.dayComparePrecise(beginDate,now);
    }
}
