package com.yeejoin.amos.fas.business.service.impl;

import com.yeejoin.amos.fas.business.dao.mapper.PlanDetailMapper;
import com.yeejoin.amos.fas.business.dao.repository.*;
import com.yeejoin.amos.fas.business.service.intfc.ContingencyPlanService;
import com.yeejoin.amos.fas.business.util.Bean;
import com.yeejoin.amos.fas.business.vo.PlanDetailVo;
import com.yeejoin.amos.fas.common.enums.ContingencyPlanStatusEnum;
import com.yeejoin.amos.fas.dao.entity.*;
import com.yeejoin.amos.fas.exception.YeeException;
import com.yeejoin.amos.feign.privilege.model.AgencyUserModel;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;

/**
 * @program: YeeAmosFireAutoSysRoot
 * @description: 预案实现类
 * @author: wujunkai
 * @create: 2021-01-14 14:59
 */
@Service
public class ContingencyPlanServiceImpl implements ContingencyPlanService {

    private final IPlanDetailDao planDetailDao;
    private final IPlanDocDao planDocDao;
    private final IPlanEquipmentDao planEquipmentDao;
    private final IPlanRuleDao planRuleDao;
    private final IPlanClassifyTreeDao classifyTreeDao;
    private final PlanDetailMapper planDetailMapper;

    @Autowired
    public ContingencyPlanServiceImpl(IPlanDetailDao planDetailDao, IPlanDocDao planDocDao, IPlanEquipmentDao planEquipmentDao,
                                      IPlanRuleDao planRuleDao, IPlanClassifyTreeDao classifyTreeDao, PlanDetailMapper planDetailMapper) {
        this.planDetailDao = planDetailDao;
        this.planDocDao = planDocDao;
        this.planEquipmentDao = planEquipmentDao;
        this.planRuleDao = planRuleDao;
        this.classifyTreeDao = classifyTreeDao;
        this.planDetailMapper = planDetailMapper;
    }

    @Override
    @Transactional(rollbackFor = {YeeException.class, Exception.class})
    public PlanDetailVo createPlan(PlanDetailVo planDetail) {
        PlanDoc planDoc = planDetail.getPlanDoc();
        PlanRule planRule = planDetail.getPlanRule();
        List<PlanEquipment> planEquipment = planDetail.getPlanEquipment();
        if (StringUtils.isBlank(planDetail.getPlanName()) || null == planDetail.getClassifyId()
                || null == planDoc || null == planDoc.getDocId()
                || null == planRule || null == planRule.getRuleId()
                || null == planEquipment || planEquipment.isEmpty()) {
            throw new YeeException("参数错误");
        }
        planEquipment.forEach(equipment -> {
            if (null == equipment || null == equipment.getFireEquipmentId()) {
                throw new YeeException("参数错误");
            }
        });
        // 状态为草稿
        planDetail.setStatus(ContingencyPlanStatusEnum.DRAFT.getCode());
        planDetail.setIsDelete(false);
        PlanDetail planEntity = new PlanDetail();
        BeanUtils.copyProperties(planDetail, planEntity);
        long planId = planDetailDao.save(planEntity).getId();
        planDetail.setId(planId);
        planDoc.setPlanId(planId);
        planDocDao.save(planDoc);
        planRule.setPlanId(planId);
        planRuleDao.save(planRule);
        planEquipment.forEach(equipment -> equipment.setPlanId(planId));
        planEquipmentDao.saveAll(planEquipment);
        return planDetail;
    }

    @Override
    public PlanDetailVo editPlan(PlanDetailVo planDetail) {
        PlanDoc planDoc = planDetail.getPlanDoc();
        PlanRule planRule = planDetail.getPlanRule();
        List<PlanEquipment> planEquipment = planDetail.getPlanEquipment();
        if (StringUtils.isBlank(planDetail.getPlanName()) || null == planDetail.getClassifyId()
                || null == planDoc || null == planDoc.getDocId()
                || null == planRule || null == planRule.getRuleId()
                || null == planEquipment || planEquipment.isEmpty()) {
            throw new YeeException("参数错误");
        }
        planEquipment.forEach(equipment -> {
            if (null == equipment || null == equipment.getFireEquipmentId()) {
                throw new YeeException("参数错误");
            }
        });
        long planId = planDetail.getId();
        PlanDetail oldPlan = planDetailDao.findById(planId).orElse(null);
        if (null == oldPlan) {
            throw new YeeException("数据不存在");
        }
        if (ContingencyPlanStatusEnum.getEnum(oldPlan.getStatus()) == ContingencyPlanStatusEnum.DRAFT
                || ContingencyPlanStatusEnum.getEnum(oldPlan.getStatus()) == ContingencyPlanStatusEnum.NOAVAILABLE) {
            // 状态设置为草稿
            planDetail.setStatus(ContingencyPlanStatusEnum.DRAFT.getCode());
        } else {
            throw new YeeException("不可编辑的状态");
        }
        PlanDetail planEntity = new PlanDetail();
        BeanUtils.copyProperties(planDetail, planEntity);
        planDetailDao.saveAndFlush(planEntity);

        planDoc.setPlanId(planId);
        planDocDao.deleteByPlanId(planId);
        planDocDao.save(planDoc);

        planRule.setPlanId(planId);
        planRuleDao.deleteByPlanId(planId);
        planRuleDao.save(planRule);

        planEquipment.forEach(equipment -> equipment.setPlanId(planId));
        planEquipmentDao.deleteByPlanId(planId);
        planEquipmentDao.saveAll(planEquipment);
        return planDetail;
    }

    @Override
    public PlanDetailVo detail(Long id) {
        PlanDetail detail = planDetailDao.findById(id).orElse(null);
        if (null == detail) {
            throw new YeeException("数据不存在");
        }
        PlanDetailVo detailVo = new PlanDetailVo();
        BeanUtils.copyProperties(detail, detailVo);
        List<PlanDoc> docs = planDocDao.getPlanDocsByPlanId(id);
        if (!docs.isEmpty()) {
            detailVo.setPlanDoc(docs.get(0));
        }
        List<PlanRule> rules = planRuleDao.getPlanDocsByPlanId(id);
        if (!rules.isEmpty()) {
            detailVo.setPlanRule(rules.get(0));
        }
        List<PlanEquipment> equipments = planEquipmentDao.getPlanDocsByPlanId(id);
        detailVo.setPlanEquipment(equipments);
        //TODO-设置执行次数
        detailVo.setExecutionTimes(0);
        PlanClassifyTree classifyTree = classifyTreeDao.findById(detailVo.getClassifyId()).orElse(null);
        if (null != classifyTree) {
            detailVo.setClassifyName(classifyTree.getClassifyName());
        }
        return detailVo;
    }

    @Override
    public List<PlanDetail> activatePlan(List<Long> idList) {
        List<PlanDetail> planDetailList = planDetailDao.getPlanDetailsByIdInAndIsDelete(idList, false);
        if (!planDetailList.isEmpty()) {
            planDetailList.forEach(plan -> {
                if (ContingencyPlanStatusEnum.getEnum(plan.getStatus()) != ContingencyPlanStatusEnum.DRAFT
                        && ContingencyPlanStatusEnum.getEnum(plan.getStatus()) != ContingencyPlanStatusEnum.AVAILABLE
                        && ContingencyPlanStatusEnum.getEnum(plan.getStatus()) != ContingencyPlanStatusEnum.NOAVAILABLE) {
                    throw new YeeException("包含不可启用的状态");
                }
                plan.setStatus(ContingencyPlanStatusEnum.AVAILABLE.getCode());
            });
            planDetailMapper.updatePlanStatusByIdList(idList, ContingencyPlanStatusEnum.AVAILABLE.getCode());
        }
        return planDetailList;
    }

    @Override
    public List<PlanDetail> deactivatePlan(List<Long> idList) {
        List<PlanDetail> planDetailList = planDetailDao.getPlanDetailsByIdInAndIsDelete(idList, false);
        if (!planDetailList.isEmpty()) {
            planDetailList.forEach(plan -> {
                if (ContingencyPlanStatusEnum.getEnum(plan.getStatus()) != ContingencyPlanStatusEnum.AVAILABLE
                        && ContingencyPlanStatusEnum.getEnum(plan.getStatus()) != ContingencyPlanStatusEnum.NOAVAILABLE) {
                    throw new YeeException("包含不可禁用的状态");
                }
                plan.setStatus(ContingencyPlanStatusEnum.NOAVAILABLE.getCode());
            });
            planDetailMapper.updatePlanStatusByIdList(idList, ContingencyPlanStatusEnum.NOAVAILABLE.getCode());
        }
        return planDetailList;
    }

    @Override
    public Boolean delete(List<Long> idList) {
        planDetailMapper.updateIsDeleteByIdList(idList, true);
        return true;
    }
}