package com.yeejoin.amos.api.tool.utils;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yeejoin.amos.api.tool.face.model.DatabaseConnectionModel;
import com.yeejoin.amos.api.tool.face.orm.entity.TableColumn;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import java.sql.*;
import java.util.*;

@Component
public class DatabaseUtils {

    /**
     * 数据库备份路径
     */
    public static final String BACKUP_PATH = "/db/";

    /**
     * 数据库备份文本前缀
     */
    public static String ONESQL_PREFIX = "";
    public static String SUFFIX = "sql";
    public static String BR = "\r\n";
    public static String SLASH = "/";
    public static String BRANCH = ";";
    private static String SPLIT = "`";
    private static String SPACE = " ";
    private static String INSERT_INTO = " INSERT INTO ";
    private static String CREATE_INTO = " CREATE TABLE ";
    private static String VALUES = "VALUES";
    private static String LEFTBRACE = "(";
    private static String RIGHTBRACE = ")";
    private static String QUOTES = "'";
    private static String COMMA = ",";
    private static String DISABLEFOREIGN = "SET FOREIGN_KEY_CHECKS = 0;\r\n";
    private static String ABLEFOREIGN = "SET FOREIGN_KEY_CHECKS = 1;\r\n";
    public static String DELIMITER = "###################################";


    private static String driverClassName;
    private static String jdbcDefaultUsername;
    private static String jdbcDefaultPassword;
    private static String jdbcDefaultIp;
    private static String jdbcDefaultPort;

    @Value("${datasource.driver-class-name}")
    public void setDriverClassName(String param) {
        driverClassName = param;
    }
    @Value("${datasource.username}")
    public void setJdbcDefaultUsername(String param) {
        jdbcDefaultUsername = param;
    }
    @Value("${datasource.password}")
    public void setJdbcDefaultPassword(String param) {
        jdbcDefaultPassword = param;
    }
    @Value("${datasource.ip}")
    public void setJdbcDefaultIp(String param) {
        jdbcDefaultIp = param;
    }
    @Value("${datasource.port}")
    public void setJdbcDefaultPort(String param) {
        jdbcDefaultPort = param;
    }



    /**
     * 连接数据库
     * @param model DataBaseLinkModel
     * @return 数据库连接
     */
    public static Connection getConnection(DatabaseConnectionModel model) {
        try {
            // 加载数据库驱动
            Class.forName(driverClassName);
            if (model.getPrefix() == null || model.getPrefix().isEmpty() || model.getPrefix().equals("null")) {
                model.setPrefix("");
            }
            if (model.getSuffix() == null || model.getSuffix().isEmpty() || model.getSuffix().equals("null")) {
                model.setSuffix("");
            }
            // 连接新增的数据库
            String dbUrl = "jdbc:mysql://" + model.getIp() + ":" + model.getPort() + "/" +
                    model.getPrefix() + model.getDbName() + model.getSuffix();
//            String dbUrl = "jdbc:mysql://" + model.getIp() + ":" + model.getPort() + "/" + model.getDbName();
            // 解密pwd，并连接
            return DriverManager.getConnection(dbUrl, model.getUserName(), AESUtils.decrypt(model.getPassWord()));

        } catch (ClassNotFoundException e) {
            // 处理数据库驱动加载异常
            e.printStackTrace();
        } catch (SQLException e) {
            // 处理数据库连接异常
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 以默认方式 获取数据源
     * @param dbName 数据库名称
     * @return 数据库连接
     */
    public static Connection getConnection(String dbName) throws Exception {
        DatabaseConnectionModel model = new DatabaseConnectionModel();
        model.setIp(jdbcDefaultIp);
        model.setPort(jdbcDefaultPort);
        model.setDbName(dbName);
        model.setUserName(jdbcDefaultUsername);
        model.setPassWord(AESUtils.encrypt(jdbcDefaultPassword));

        return getConnection(model);
    }

    /**
     * 获取数据库下的所有表名
     */
    public static List<String> getTableNames(String dbName, Connection connection) throws SQLException {
        String sql = "select table_name from information_schema.tables where table_schema='" + dbName + "';";
        return getList(sql, connection);
    }

    /**
     * 测试数据库连接是否成功
     */
    public static boolean isConnectionValid(DatabaseConnectionModel model) throws Exception {
        Connection connection = DatabaseUtils.getConnection(model);
        if (connection != null) {
            connection.close();
            return true;
        }
        return false;
    }

    /**
     * 获取所有数据库名称，如果为空，则返回空的List<String>
     */
    public static List<String> getAllDatabaseNames(Connection connection) throws SQLException {
        return new ArrayList<>(DatabaseUtils.getList("SHOW DATABASES;", connection));
    }

    /**
     * 获取maps
     */
    public static List<Map<String, Object>> getMaps(String sql, Connection connection) throws SQLException {
        Statement statement = connection.createStatement();
        ResultSet rs = null;
        List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
        try {
            rs = statement.executeQuery(sql);
            ResultSetMetaData md = rs.getMetaData();
            int columnCount = md.getColumnCount();
            while (rs.next()) {
                Map<String, Object> rowData = new HashMap<String, Object>();
                for (int i = 1; i <= columnCount; i++) {
                    rowData.put(md.getColumnName(i), rs.getObject(i));
                }
                list.add(rowData);
            }
        } catch (SQLException e) {
//            e.printStackTrace();
            return list;
        } finally {
            try {
                if (rs != null)
                    rs.close();
                rs = null;
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        return list;
    }


    public static List<String> getList(String sql, Connection connection) throws SQLException {
        Statement statement = connection.createStatement();
        ResultSet rs = statement.executeQuery(sql);
        List<String> list = new ArrayList<>();
        try {
            ResultSetMetaData md = rs.getMetaData();
            int columnCount = md.getColumnCount();
            while (rs.next()) {
                for (int i = 1; i <= columnCount; i++) {
                    list.add(rs.getObject(i).toString());
                }
            }
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            try {
                if (rs != null)
                    rs.close();
                rs = null;
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        return list;
    }

    /**
     * 获取表单列名
     */
    public static List<TableColumn> getTableColumn(String tableName, Connection connection) throws SQLException {
        String columnSql = "" +
                "\t\t SELECT\n" +
                "\t\t table_name as table_name,\n" +
                "\t\t\tcolumn_name as columnName,\n" +
                "\t\t\t( CASE WHEN ( is_nullable = 'no' && column_key != 'PRI' ) THEN '1' ELSE NULL END ) AS isRequired,\n" +
                "\t\t\t( CASE WHEN column_key = 'PRI' THEN '1' ELSE '0' END ) AS isPk,\n" +
                "\t\t\tordinal_position AS sort,\n" +
                "\t\t\tcolumn_comment as columnComment,\n" +
                "\t\t\t( CASE WHEN extra = 'auto_increment' THEN '1' ELSE '0' END ) AS isIncrement,\n" +
                "\t\t\tcolumn_type as columnType \n" +
                "\t\tFROM\n" +
                "\t\t\tinformation_schema.COLUMNS \n" +
                "\t\tWHERE\n" +
                "\t\t\ttable_schema = database()\n" +
                "\t\t\tAND table_name = \"" + tableName + "\"\n" +
                "\t\tORDER BY\n" +
                "\t\t\tordinal_position\t";
        List<Map<String, Object>> columnList = DatabaseUtils.getMaps(columnSql, connection);
        List<TableColumn> columns = new ArrayList<>();
        for (Map<String, Object> map : columnList) {
            TableColumn column = JSON.parseObject(JSON.toJSONString(map), TableColumn.class);
            if (!column.getColumnName().equals("content"))
                columns.add(column);
        }
        return columns;
    }

    /**
     * 获取insert语句
     */

    public static StringBuffer getInsertSQL(List<Map<String, Object>> dataList, List<TableColumn> columns, String tableName) {
        StringBuffer sql = new StringBuffer();
        for (Map<String, Object> map : dataList) {
            sql.append(INSERT_INTO);
            sql.append(SPACE).append(tableName).append(SPACE);
            sql.append(LEFTBRACE);
            for (TableColumn column : columns) {
                sql.append(SPLIT);
                sql.append(column.getColumnName());
                sql.append(SPLIT);
                sql.append(COMMA);
            }
            sql.deleteCharAt(sql.length() - 1);
            sql.append(RIGHTBRACE);
            sql.append(VALUES);
            sql.append(LEFTBRACE);
            for (TableColumn column : columns) {
                if (column.getColumnType().contains("bigint") ||
                        column.getColumnType().contains("int") ||
                        column.getColumnType().contains("bit(1)")) {
                    if (!ObjectUtils.isEmpty(map.get(column.getColumnName()))) {
                        sql.append(map.get(column.getColumnName()));
                    } else {
                        sql.append("0");
                    }
                } else {
                    if (!ObjectUtils.isEmpty(map.get(column.getColumnName()))) {
                        if (column.getColumnName().equals("UIOBJECT") || column.getColumnName().equals("CONTENT")){
                            int start = sql.length()+2;
                            sql.append("'").append(JSON.toJSONString(map.get(column.getColumnName())).
                                    replaceAll("'","\\\\'")).append("'");
                            int end = sql.length()-2;
                            sql.delete(start-1,start);
                            sql.delete(end-1,end);
                        }else if (column.getColumnName().equals("PUBLISH_CODE")){
                            sql.append("'").append((map.get(column.getColumnName())).toString().replaceAll("'","\\\\'")).append("'");
                        }else {
                            sql.append("'").append(map.get(column.getColumnName())).append("'");
                        }
                    } else {
                        sql.append("null");
                    }
                }
                sql.append(COMMA);
            }
            sql.deleteCharAt(sql.length() - 1);
            sql.append(RIGHTBRACE);
            sql.append(BRANCH).append(BR);
        }
        return sql;
    }



    public static <V> Page<V> listToPage(List<V> list, int current, int size) {
        //当前页
        Page<V> page = new Page<>(current, size);
        int count = list.size();
        List<V> pageList = new ArrayList<>();
        //计算当前页第一条数据的下标
        int currId = current > 1 ? (current - 1) * size : 0;
        for (int i = 0; i < size && i < count - currId; i++) {
            pageList.add(list.get(currId + i));
        }
        page.setSize(size);
        page.setCurrent(current);
        page.setTotal(count);
        //计算分页总页数
        page.setPages(count % 10 == 0 ? count / 10 : count / 10 + 1);
        page.setRecords(pageList);
        return page;
    }
}
