import React, { Component } from 'react';
import { PieChart } from 'amos-viz/lib/echarts';
import momentDateUtil from '../../common/utils/momentDateUtil';
import { ScreenCard, withSubscribeMQTT } from './../../common';
import {
  alarmPercentAction
} from './../api';

/**
 * 警情类型占比
 *
 * @class AlarmTypeScale
 * @extends {Component}
 */
@withSubscribeMQTT({ topic: 'alertNotice' })
class AlarmTypeScale extends Component {
  static propTypes = {

  }

  constructor(props) {
    super(props);
    this.state = {
      alarmTypeScaleData: [],
      totalName: '',
      totalValue: ''
    };
  }

  componentDidMount() {
    this.onRequest();
  }

  onRequest = () => {
    const { beginDate, endDate } = momentDateUtil.getCurrentYearDays();
    let alarmTypeScaleData = [];
    alarmPercentAction(beginDate,endDate).then(d => {
      if (d) {
        alarmTypeScaleData = [
          {
            name: '故障',
            value: d.faultAlarm ? d.faultAlarm : 0
          },
          {
            name: '火警',
            value: d.fireAlarm ? d.fireAlarm : 0
          },
          {
            name: '屏蔽',
            value: d.shieldAlarm ? d.shieldAlarm : 0
          },
          {
            name: '漏油',
            value: d.oilLeakAlarm ? d.oilLeakAlarm : 0
          },
          {
            name: '告警',
            value: d.alarm ? d.alarm : 0
          }
        ];

        this.setState({ alarmTypeScaleData }, () => this.getTotal());
      }
    });
  }

  getTotal = () => {
    const { alarmTypeScaleData } = this.state;
    let totalValue = 0;
    alarmTypeScaleData.forEach(item => {
      totalValue += item.value;
    });
    this.setState({ totalValue, totalName: '总数' });
  }

  /**
   * 图例展示形式
   */
  legendStyle = (name) => {
    const { alarmTypeScaleData } = this.state;
    let total = 0;
    let tarValue;
    for (let i = 0; i < alarmTypeScaleData.length; i++) {
      total += Number(alarmTypeScaleData[i].value);
      if (alarmTypeScaleData[i].name === name) {
        tarValue = alarmTypeScaleData[i].value;
      }
    }
    let v = tarValue;
    // eslint-disable-next-line no-magic-numbers
    let p = Math.round((tarValue / total) * 100);
    return `${name} ${v}  (${p}%)`;
  }

  getPieOption = () => {
    const { alarmTypeScaleData, totalName, totalValue } = this.state;
    return {
      color: ['#DBC521', '#EE6328', '#54DBFF', '#3B9DFF', '#C155E6'],
      title: {
        text: totalValue,
        subtext: totalName,
        left: '31%',
        textStyle: {
          fontSize: 30,
          color: '#fff'
        },
        subtextStyle: {
          fontSize: 20,
          color: '#fff'
        },
        top: '40%',
        textAlign: 'center',
        itemGap: 1
      },
      tooltip: {
        show: true
      },
      legend: {
        orient: 'vertical',
        left: '60%',  // 图例距离左的距离
        y: 'center',  // 图例上下居中
        data: ['故障','火警','屏蔽','漏油','告警'],
        itemGap: 25,
        textStyle: {
          color: '#fft'
        },
        formatter: (name) => this.legendStyle(name)
      },
      series: [
        {
          type: 'pie',
          data: alarmTypeScaleData,
          radius: ['40%', '60%'],
          labelLine: {
            normal: {
              show: false
            }
          },
          label: {
            normal: {
              show: false
            }
          },
          center: ['32%', '50%'] // 图的位置，距离左跟上的位置
        }
      ]
    };
  }

  render() {
    return (
      <ScreenCard title="警情类型占比">
        <div className="ccs-iot-monitoring-right-chart">
          <PieChart option={this.getPieOption()} />
        </div>
      </ScreenCard>
    );
  }
}

export default AlarmTypeScale;
