import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { StdTable } from 'amos-designer';

/**
 * TableList
 *
 * @class TableList
 * @extends {Component}
 */
class TableList extends Component {

  static propTypes = {
    dataSource: PropTypes.array,
    columns: PropTypes.array,
    rowKey: PropTypes.string,
    total: PropTypes.number,
    pageSize: PropTypes.number,
    current: PropTypes.number,
    onPageChange: PropTypes.func,
    rowSelection: PropTypes.any,
    selectedRowKeys: PropTypes.array,
    onSelectChange: PropTypes.func,
    onSorter: PropTypes.func,
    sortedInfo: PropTypes.object,
    tableParams: PropTypes.object,
    pagination: PropTypes.bool,
    onRowClick: PropTypes.func,
    rowClassName: PropTypes.string
  };

  static defaultProps = {
    selectedRowKeys: [],
    sortedInfo: {}
  };

  constructor(props) {
    super(props);
    this.state = {

    };
  }

  componentDidMount() {
    this.onResize();
    window.addEventListener('resize', this.onResize);
  }

  componentDidUpdate(preProps) {
    if (preProps.dataSource.length !== this.props.dataSource.length) {
      this.onResize();
    }
  }

  componentWillUnmount() {
    window.removeEventListener('resize', this.onResize);
  }

  onResize = () => {
    const headHeight = 70;
    const tableBody = this.table.getElementsByClassName('amos-table-tbody')[0];
    if (tableBody.scrollHeight > (this.table.clientHeight - headHeight)) {
      this.setState({ height: this.table.clientHeight - headHeight });
    } else {
      this.setState({ height: 0 });
    }
  }

  onPageSize = (page, pageSize) => {
    const { onPageChange, current } = this.props;
    let newPage = current;
    let newPageSize = this.props.pageSize;
    if (page !== undefined) {
      newPage = page;
    }
    if (pageSize !== undefined) {
      newPageSize = pageSize;
    }
    onPageChange && onPageChange(newPage, newPageSize);
  }

  handleChange = (pagination, filters, sorter) => {
    this.props.onSorter(sorter);
  }

  getColumns = () => {
    const { columns } = this.props;
    return columns;
  }

  render() {
    let { dataSource, tableParams, rowKey, rowSelection, selectedRowKeys, onSelectChange,
      total, pageSize, current, pagination, onRowClick, rowClassName } = this.props;
    const { height } = this.state;
    if (rowSelection) {
      rowSelection = {
        ...rowSelection,
        selectedRowKeys,
        onChange: onSelectChange
      };
    }
    return (
      <div className="gm-tablepage-table" ref={node => this.table = node}>
        <StdTable
          columns={this.getColumns()}
          rowKey={rowKey || 'id'}
          dataSource={dataSource}
          onChange={this.handleChange}
          pagination={pagination ? {
            total,
            pageSizeProps: { popoverClassName: 'gm-tablepage-popover' },
            showTotal: e => `共${e}条`,
            pageSize,
            current,
            showQuickJumper: true,
            showSizeChanger: true,
            onShowSizeChange: this.onPageSize,
            onChange: this.onPageSize
          } : false}
          scroll={{ y: height }}
          rowSelection={rowSelection}
          {...tableParams || {}}
          onRowClick={onRowClick}
          rowClassName={rowClassName}
        />
      </div>
    );
  }
}

export default TableList;
