import { IHub } from 'amos-viz/lib/widgets/hub';
import { Store } from 'amos-tool';

const lsTool = Store.lsTool;

export const hub = new IHub();


/**
 * 页面事件常量
 */
export const eventConsts = {
  SWITCH_WHOLE_COUNTRY_STATION: 'switch_whole_country_station',
  STATION_CODE: 'station_code',
  FIRE_ALARM_CHANGE: 'fire_alarm_change',
  ALARM_INFO: 'alarm_info', // 告警选中
  EMERGENCY_MODEL: 'emergency_model'
};


const storageFlag = '@message_'; // localStorage 消息键值标识
const mapFlag = '@map_'; // 地图 消息键值标识

export class MessageEntity {
  constructor() {
    this.mapIFrame = null;
    this.sendMapMesssageEntity = null;
    this.mapCache = [];
    const originalSetItem = localStorage.setItem;
    localStorage.setItem = function(key,value){
      const setItemEvent = new window.Event('setItemEvent');
      setItemEvent.newValue = JSON.stringify({ key, value });
      window.dispatchEvent(setItemEvent);
      originalSetItem.apply(this, [key,value]);
    };
  }

  setSendMapMesssageEntity(method) {
    this.sendMapMesssageEntity = method;
    if (this.mapCache.length > 0) {
      console.log('缓存的未发送消息', this.mapCache);
      this.mapCache.forEach(e => {
        this.mapTrigger(e.topic, e.data);
      });
    }
  }

  /**
   * 监听消息（可以同时监听与地图、另一个屏和其他页面间的消息）
   * @param {*} topic 消息主题
   * @param {*} callback 监听函数
   * @param {*} onMap 是否监听地图发来的消息
   * @param {*} onOtherScreen 是否监听其他屏幕发来的消息
   * @memberof Message
   */
  on(topic, callback, onMap, onOtherScreen) {
    console.log('监听其他屏幕发来的消息');
    console.log(topic);
    hub.event.on(topic, callback);
    if (onMap) {
      this.mapSubscribe(topic, callback);
    }
    if (onOtherScreen) {
      this.storageSubscribe(topic, callback);
    }
  }
  /**
   * 监听其他屏幕发来的消息
   * @param {*} topic 消息主题
   * @param {*} callback 监听函数
   */
  storageSubscribe(topic, callback) {
    window.addEventListener('storage', (e) => {
      if (e.key) {
        try {
          const mesage = JSON.parse(e.newValue);
          if (e.key && e.key.startsWith(storageFlag)) {
            if (`${storageFlag}${topic}` === e.key) {
              callback(mesage);
            }
          }
        } catch (error) {
          console.log();
        }
      }
    });
    window.addEventListener('setItemEvent', (e) => {
      if (e.newValue) {
        try {
          const data = JSON.parse(e.newValue);
          if (data.key && data.key.startsWith(storageFlag)) {
            if (`${storageFlag}${topic}` === data.key) {
              callback(data.value);
            }
          }
        } catch (error) {
          console.log();
        }
      }
    });
  }

  /**
   * 监听地图发来的消息
   * @param {*} topic  消息主题
   * @param {*} callback  监听函数
   */
  mapSubscribe(topic, callback) {
    window.addEventListener('message', (e) => {
      if (e.data) {
        try {
          const messagData = JSON.parse(e.data);
          const { topic: _topic, data } = messagData || {};
          if (_topic && _topic.startsWith(mapFlag)) {
            if (`${mapFlag}${topic}` === _topic) {
              callback(data);
            }
          }
        } catch (error) {
          console.log();
        }
      }
    }, false);
  }

  off(topic) {
    hub.event.off(topic);
  }
  /**
   * 发送消息（可以同时向地图、另一个屏和其他页面发消息）
   * @param {String} topic 消息主题
   * @param {*} data 消息数据
   * @param {Boolean} emitOtherScreen 同步发送到其他屏幕
   */
  emit(topic, data, emitOtherScreen) {
    hub.event.emit(topic, data);
    if (emitOtherScreen) {
      this.storageTrigger(topic, data);
    }
  }
  /**
   * 向其他屏幕发送消息
   * @param {*} topic 消息主题
   * @param {*} data 消息数据
   */
  storageTrigger(topic, data) {
    let messageData = { topic: `${storageFlag}${topic}`, data };
    messageData = JSON.stringify(data);
    // lsTool.write(`${storageFlag}${topic}`, messageData);
    localStorage.setItem(`${storageFlag}${topic}`, messageData);
  }

  getStorageData(topic) {
    console.log(topic);
    const data = lsTool.read(`${storageFlag}${topic}`);
    let result = null;
    try {
      result = JSON.parse(data);
    } catch (error) {
      result = data;
    }
    return result;
  }
  /**
   * 向地图发送消息
   * @param {*} topic 消息主题
   * @param {*} data 消息数据
   */
  mapTrigger(topic, data) {
    if (this.sendMapMesssageEntity) {
      this.sendMapMesssageEntity({ topic: `@map_${topic}`, data });
    } else {
      this.mapCache.push({ topic, data });
    }
  }
}

export const message = new MessageEntity();
