import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Form, Input, PortalSidePane } from 'amos-framework';
import { formLayout, Btn, VideoMatrix, FireFilter } from '@gm/graphmod-common';
import { tools } from '@gm/graphmod-utils';
import ImageConsts from '@gm/graphmod-assets';
import VedioDetailsPane from './VedioDetailsPane';
import { vedioPageListAction } from './../api';

const FormItem = Form.Item;
/**
 * 消防设备
 *
 * @class MonitorContent
 * @extends {Component}
 */
class MonitorContent extends Component {
  static propTypes = {
    oauthConfig: PropTypes.object, // 权限配置
    dataConfig: PropTypes.object, // 权限配置
    routeState: PropTypes.object,
    backdata: PropTypes.object,
    buildObj: PropTypes.object, // 选择的设备类型
    code: PropTypes.string, // 换流站code
    stationData: PropTypes.object
  };

  static contextTypes = {
    browserHistory: PropTypes.object
  };

  static defaultProps = {
    routeState: {},
    dataConfig: {},
    buildObj: {},
    oauthConfig: {}
  };

  constructor(props) {
    super(props);
    this.state = {
      videos: [], // 视频信息列表
      total: 0, // 总数
      selectedRowKeys: [], // 选择的数据id集合
      selectedRows: [],
      page: 1, // 当前页
      size: 9, // 页码大小
      filters: {}, // 过滤条件
      info: {}, // 右侧详情信息对象
      showFilter: false,
      showDetails: false
    };
  }

  componentDidMount() {
    this.onRequestTableData();
  }

  componentDidUpdate(prevProps) {
    if (this.props.buildObj.id !== prevProps.buildObj.id) {
      this.onRequestTableData(true);
    }
  }

  componentWillUnmount() {
    this.setState({ showFilter: false, showDetails: false });
  }

  /**
   * 请求数据
   * @memberof FireSystem
   */
  onRequestTableData = init => {
    let { code, buildObj } = this.props;
    const { page, size, filters } = this.state;
    let buildingMrids = '';
    if (buildObj.type === 'all') {
      code = '-1';
    } else if (buildObj.type === 'station') {
      code = buildObj.id;
    } else {
      buildingMrids = buildObj.treeMrids || '';
    }
    init && this.setState({ page: 1 });
    vedioPageListAction(init ? 1 : page, size, filters.name || '', filters.mrid || '', code || '-1', buildingMrids).then(d => {
      const date =  d.records.map((item) => {
        let json = { ...item };
        json.key = item.id;
        json.vedioId = item.mrid;
        json.token = item.token || '';
        json.url = item.playUrl;
        json.code = item.mrid;
        return json;
      });
      this.setState({
        videos: date,
        total: Number(d.total),
        page: d.current
      });
    });
  };

  /**
   * 接口获取右侧详情信息
   * @memberof FireSystem
   */
  onRequestInfo = data => {
    this.setState({ info: data });
  };

  /**
   * 隐藏显示左侧树
   *
   * @memberof FireSystem
   */
  handleHideGroup = () => {};
  /**
   * 过滤条件改变
   *
   * @memberof FireSystem
   */
  handleFilterChange = filters => {
    this.setState({ filters });
  };

  /**
   * 处理分页
   * @memberof FireSystem
   */
  handlePageChange = (page, size) => {
    this.setState({ videos: [] });
    this.setState({ page, size }, this.onRequestTableData);
  };
  /**
   * 处理查询
   * @memberof FireSystem
   */
  handleSearch = () => {
    this.onRequestTableData();
  };
  /**
   * 处理重置
   * @memberof FireSystem
   */
  handleReset = () => {
    this.setState({ filters: {} }, this.onRequestTableData);
  };

  handleShowFilter = () => {
    this.setState({
      showFilter: !this.state.showFilter,
      showDetails: !this.state.showFilter ? false : this.state.showDetails
    });
  };
  /**
   * 点击某个监控视屏
   * @param {*} vedio
   */
  handleVedioClick = (vedio, event) => {
    event && event.stopPropagation();
    if (vedio) {
      this.setState({
        info: vedio || {},
        showDetails: true,
        showFilter: false
      });
    } else {
      this.setState({
        info: {},
        showDetails: false
      });
    }
  };

  /**
   * 过滤条件项
   * @memberof FireSystem
   */
  renderFilters = (onChange, filters) => {
    return (
      // eslint-disable-next-line react/jsx-fragments
      <React.Fragment>
        <FormItem className={formLayout('100', '100')} label={<span>摄像头名称</span>} field="name">
          <Input value={filters.name} onChange={e => onChange('name', e.target.value)} />
        </FormItem>
        <FormItem className={formLayout('100', '100')} label={<span>摄像头编码</span>} field="mrid">
          <Input value={filters.mrid} onChange={e => onChange('mrid', e.target.value)} />
        </FormItem>
      </React.Fragment>
    );
  };

  /**
   * 关闭右侧详情
   */
  closeDetails = () => {
    this.setState({
      showDetails: !this.state.showDetails
    });
  }

  render() {
    const { videos, total, page, size, showFilter, showDetails, filters, info } = this.state;
    return (
      <div className="ccs-com-firepanal" ref={node => this.root = node}>
        <div className="ccs-com-firepanal-header">
          <div style={{ flex: '1' }} />
          <div className="ccs-com-firepanal-rightbtns">
            <VideoMatrix.Pagination page={page} size={size} onPageChange={this.handlePageChange} total={total} />
            <Btn label="筛选" onClick={this.handleShowFilter} icon={ImageConsts.common.filter} />
          </div>
        </div>
        <div
          className="ccs-com-firepanal-content"
          id="fireVideoMonitorContent"
          onClick={tools.isClickBlankByClass(() => this.handleVedioClick(), ['mod-com-videomatrix-item', 'ccs-vediodetails-pane'])}
        >
          <div ref={node => this.a = node} />
          <div ref={node => this.b = node} />
          <VideoMatrix
            videos={videos}
            matrix={size}
            title={e => `${e.position}-${e.name || ''}${e.mrid ? `(${e.mrid})` : ''}`}
            // title={(e) => `${e.location ? e.address ? `${e.location}-${e.address}` : e.location : e.address}-${e.name || ''}${e.code ? `(${e.code})` : ''}`}
            more={this.handleVedioClick}
            dataConfig={this.props.dataConfig}
          />
          <PortalSidePane
            top={0}
            left={null}
            right={0}
            visible={showFilter}
            animateName="slide-right"
            getContainer={() => this.a}
            animateProps={{ transitionAppear: true }}
            animateDelay={0}
          >
            <div style={{ height: '100%', width: 240 }}>
              <FireFilter
                className="mods-com-firepanal-filter"
                onShowFilter={this.handleShowFilter}
                show={showFilter}
                filters={filters}
                onFilterChange={this.handleFilterChange}
                onSearch={this.handleSearch}
                onReset={this.handleReset}
                renderFilters={this.renderFilters}
              />
            </div>
          </PortalSidePane>
          {
            // <PortalSidePane
            //   top={0}
            //   left={null}
            //   right={0}
            //   animateName="slide-right"
            //   getContainer={() => this.b}
            //   visible={showDetails}
            //   animateProps={{ transitionAppear: true }}
            //   animateDelay={0}
            // >
            //   <div style={{ height: '100%', width: 240 }}>
            //     <VedioDetailsPane info={info} closeDetails={this.closeDetails} />
            //   </div>
            // </PortalSidePane>
          }
        </div>
      </div>
    );
  }
}

export default MonitorContent;
