import React, { Component } from 'react';
import { canvas2img } from 'amos-tool';
import { Select, Form, Slider } from 'amos-framework';
import QRCode from 'amos-viz/lib/qrcode';
import Modal from './../modal';
import { formLayout } from './../form/FormUtils';
import './DownLoadQrCode.scss';


const Option = Select.Option;
const FormItem = Form.Item;
const imgTypes = {
  'png': 'image/png',
  'jpeg': 'image/jpeg',
  'bmp': 'image/bmp'
};
/**
 * 二维码下载
 *
 * @class DownLoadQrCode
 * @extends {Component}
 */
class DownLoadQrCode extends Component {

  constructor(props) {
    super(props);
    this.state = {
      visible: false,
      form: {
        type: 'png',
        size: 200
      },
      value: '',
      title: '',
      imgName: ''
    };
  }
  /**
   * 打开弹框并初始化数据
   * @param {*} qrCode 二维码值
   * @param {*} title 弹框标题
   * @param {*} imgName 下载图片名称
   * @memberof DownLoadQrCode
   */
  onOpen = (qrCode, title, imgName) => {
    this.setState({ visible: true, imgName, value: qrCode, title });
  }

  onSetValue = (value) => {
    this.setState({ value });
  }

  onDownload = () => {
    if (!this.state.value) {
      return;
    }
    const { form, imgName } = this.state;
    canvas2img.convert2Blob(this.qr.getElementsByTagName('CANVAS')[0], (blob) => {
      const downUrl = window.URL.createObjectURL(blob);
      const a = window.document.createElement('a');
      a.href = downUrl;
      a.download = `${imgName || '二维码'}.${form.type}`;
      a.click();
      window.URL.revokeObjectURL(downUrl);
    }, {
      type: imgTypes[form.type] || imgTypes.png,
      width: form.size,
      height: form.size,
      encoderOptions: 1
    });
    this.onCancel();
  }

  onCancel = () => {
    if (!this.state.value) {
      return;
    }
    this.setState({
      visible: false,
      form: {
        type: 'png',
        size: 200
      },
      value: '',
      title: ''
    });
  }

  onChange = (varName, value) => {
    const { form } = this.state;
    form[varName] = value;
    this.setState({ form });
  }

  render() {
    const { visible, form, value, title } = this.state;
    const defaultOptions = {
      value,
      size: form.size,
      fgColor: '#000000',
      bgColor: '#ffffff',
      level: 'L',
      tag: 'svg',
      includeMargin: false
    };
    return (
      <Modal
        modalClass="mods-com-qrdownload-wrapper"
        width={500}
        visible={visible}
        onCancel={this.onCancel}
        onOk={this.onDownload}
        okText="下载"
        title={`${title}下载` || '下载二维码'}
      >
        <div className="mods-com-qrdownload-container">
          <div className="mods-com-qrdownload-img" ref={node => this.qr = node} >
            <QRCode {...defaultOptions} />
          </div>
          <div className="mods-com-qrdownload-setting">
            <Form className="mods-com-form-tablelist-filter-content" rules={{}} ref={e => this.form = e} model={form}>
              <FormItem className={formLayout('50', '4')} label={<span>格式</span>} field="type">
                <Select value={form.type} onChange={(e) => this.onChange('type', e)}>
                  {
                    Object.keys(imgTypes).map(e => (
                      <Option key={e} value={e}>{e}</Option>
                    ))
                  }
                </Select>
              </FormItem>
              <FormItem className={formLayout('50', '4')} label={<span>大小</span>} field="size">
                <Slider value={form.size} onChange={(e) => this.onChange('size', e)} min={30} max={400}  />
              </FormItem>
            </Form>
          </div>
        </div>
      </Modal>
    );
  }
}

export default DownLoadQrCode;
