import React, { Component } from 'react';
import PropTypes from 'prop-types';
import QRCode from 'amos-viz/lib/qrcode';
import classnames from 'classnames';
import { Menu, Popover } from 'amos-framework';
import ImageConsts from '@gm/graphmod-assets';
import './FireCard.scss';

/**
 * 消防卡片
 *
 * @class FireCard
 * @extends {Component}
 */
class FireCard extends Component {
  static propTypes = {
    className: PropTypes.string, // css class
    icon: PropTypes.any,
    isSelect: PropTypes.bool,
    onClick: PropTypes.func,
    onDoubleClick: PropTypes.func,
    configs: PropTypes.array,
    qrCodeValue: PropTypes.string,
    qrCode: PropTypes.string,
    menus: PropTypes.array,
    defaultIcon: PropTypes.any,
    leftOther: PropTypes.any,
    rightOther: PropTypes.any,
    name: PropTypes.any,
    status: PropTypes.array,
    data: PropTypes.any,
    bottomIcons: PropTypes.array
  };

  static defaultProps = {
    onClick() {},
    onDoubleClick() {},
    configs: [],
    menus: [],
    defaultIcon: ImageConsts.common.defaultCard,
    status: [],
    bottomIcons: []
  };

  constructor(props) {
    super(props);
    this.state = {
      open: false
    };
  }

  onToggle = open => {
    this.setState({ open });
  };

  renderMenu = (data, menus) => {
    return (
      <Menu className="mods-com-firecard-menulist">
        {menus
          .filter(e => !e.hide)
          .map(m => {
            return (
              <Menu.Item
                key={m.key}
                onClick={e => {
                  e.stopPropagation();
                  this.onToggle(false);
                  m.click && m.click(data);
                }}
                className={classnames({ 'mods-com-firecard-menudivider': m.showDivider })}
              >
                <img src={m.icon} alt="" />
                {m.text}
              </Menu.Item>
            );
          })}
      </Menu>
    );
  };

  renderQrCode = () => {
    const { qrCodeValue, qrCode } = this.props;
    const defaultOptions = {
      value: qrCode,
      size: 96,
      fgColor: '#000000',
      bgColor: '#ffffff',
      level: 'L',
      tag: 'svg',
      includeMargin: false
    };
    return (
      <div className="mods-com-firecard-qrcode-content">
        <QRCode {...defaultOptions} />
        {qrCodeValue && <div className="mods-com-firecard-qrcode-value">{qrCodeValue}</div>}
      </div>
    );
  };

  render() {
    const { icon, configs, data, onClick, onDoubleClick, name, menus, defaultIcon, isSelect, status, bottomIcons, qrCode, leftOther, rightOther } = this.props;
    return (

      <div
        className={classnames('mods-com-firecard', { 'mods-com-firecard-selected': isSelect })}
        onClick={onClick} onDoubleClick={onDoubleClick}>

        <div className="mods-com-firecard-top">
          <div className="mods-com-firecard-icon">
            <img src={icon || defaultIcon} alt="" />
          </div>
          <div className="mods-com-firecard-info">
            <div className="mods-com-firecard-name" title={name}>{name}</div>
            {configs.map((e, index) => {
              if (e.hide) {
                if (e.hide(data)) {
                  return null;
                }
              }
              return (
                <div key={e.key} className="mods-com-firecard-infoitem">
                  {!e.hideTitle && <span className="mods-com-firecard-label">{e.title}：</span>}
                  <span
                    title={!e.render ? data[e.key] : ''}
                    className="mods-com-firecard-value"
                    style={{ whiteSpace: e.noEllipsis ? 'break-spaces' : 'nowrap' }}
                  >
                    {e.render ? e.render(data[e.key], data, index) : data[e.key]}
                  </span>
                </div>
              );
            })}
          </div>
        </div>

        <div className="mods-com-firecard-floor">
          <div className="mods-com-firecard-status">
            {qrCode && (
              <Popover
                content={this.renderQrCode()}
                contentWrapClassName="mods-com-firecard-popover"
                className="mods-com-firecard-popover-wrapper"
                direction="down"
                align="left"
                triggerMode="click"
              >
                <div className="mods-com-firecard-qrcode">
                  <img src={ImageConsts.common.qrcode} alt="" />
                </div>
              </Popover>
            )}
            {leftOther}

            {bottomIcons.map(e => {
              return e.hide ? (
                <div key={e.key} className="mods-com-firecard-icontag">
                  <img src={e.img} alt="" />
                </div>
              ) : null;
            })}

            {status.map(e => {
              if (e.hide) {
                if (e.hide(data)) {
                  return null;
                }
              }
              return e.type === 'icon' ?
                e.getImage ? (
                  <div key={e.key} className="mods-com-firecard-icontag">
                    <img src={e.getImage(data[e.key])} alt="" />
                  </div>
                ) : (
                  <span />
                )
                : (
                  <div key={e.key} className="mods-com-firecard-texttag" style={{ background: e.getColor ? e.getColor(data[e.key]) : '#00c20c' }}>
                    { e.getText ? e.getText(data) : data[e.key]}
                  </div>
                );
            })}
            {rightOther}
          </div>
          {menus && menus.length > 0 && menus.filter(e => !e.hide).length > 0 && (
            <div className="mods-com-firecard-menus">
              <Popover
                content={this.renderMenu(data, menus)}
                contentWrapClassName="mods-com-firecard-popover"
                className="mods-com-firecard-popover-wrapper"
                direction="down"
                open={this.state.open}
                onToggle={this.onToggle}
                align="right"
                triggerMode="click"
              >
                <img src={ImageConsts.common.cardMenus} alt="" />
              </Popover>
            </div>
          )}
        </div>
      </div>

    );
  }
}

export default FireCard;
