import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { LineChart } from 'amos-viz/lib/echarts';
import { ScreenCard, withSubscribeMQTT } from './../../common';
import momentDateUtil from '../../common/utils/momentDateUtil';
import {
  alarmDangerLineAction
} from './../api';

/**
 * 近七日告警隐患
 * @class AlarmDangerLine
 * @extends {Component}
 */
@withSubscribeMQTT({ topic: 'power' })
class AlarmDangerLine extends Component {
  constructor(props) {
    super(props);
    this.state = {
      dayList: [], // 前端生成最近一个月日期 x轴
      yData: []
    };
  }

  componentDidMount() {
    this.onRequest();
  }

  onRequest = () => {
    const { beginDate, endDate } = momentDateUtil.getLast7Days();
    const dayList = momentDateUtil.enumerateDaysBetweenDates(beginDate, endDate);
    let yData = [];
    alarmDangerLineAction(beginDate, endDate).then(d => {
      if (d) {
        yData = [
          {
            name: '告警',
            data: d.alarmTrend ? d.alarmTrend : [],
            type: 'line',
            smooth: true
          },
          {
            name: '隐患',
            data: d.dangerTrend ? d.dangerTrend : [],
            type: 'line',
            smooth: true
          }
        ];
        this.setState({ yData });
      }
    });
    this.setState({
      dayList
    });
  }

  getPieOption = () => {
    const { dayList, yData } = this.state;
    return {
      color: ['yellow', 'red'],
      grid: {
        x: 40,
        y: 40,
        x2: 50,
        y2: 40,
        borderWidth: 5
      },
      tooltip: {
        trigger: 'axis'
      },
      legend: {
        data: ['告警', '隐患'],
        bottom: '88%',
        textStyle: {
          color: '#fff'
        }
      },
      xAxis: {
        name: '日期',
        type: 'category',
        boundaryGap: false,
        data: dayList,
        axisPointer: {
          type: 'line'
        },
        axisLabel: {
          margin: 10,
          color: '#fff',
          fontSize: 12,
          rotate: -30
        },
        axisLine: {
          lineStyle: {
            color: '#fff'
          }
        },
        axisTick: {
          alignWithLabel: true,
          color: '#fff',
          lineStyle: {
            color: '#fff'
          }
        }
      },
      yAxis: {
        name: '数目',
        type: 'value',
        axisLabel: {
          margin: 10,
          color: '#fff',
          fontSize: 16
        },
        axisLine: {
          lineStyle: {
            color: '#fff'
          }
        },
        splitLine: {
          show: true,
          lineStyle: {
            type: 'dashed'
          }
        }
      },
      series: yData
    };
  }

  render() {
    return (
      <ScreenCard title="近7日告警隐患">
        <div className="ccs-iot-monitoring-left-chart">
          <LineChart
            option={this.getPieOption()}
          />
        </div>
      </ScreenCard>
    );
  }
}

AlarmDangerLine.propTypes = {
  messageFlag: PropTypes.number
};

export default AlarmDangerLine;
