/* eslint-disable no-magic-numbers */
import classnames from 'classnames';
import { trim } from 'amos-tool';

/**
 * form表单布局样式设置方法
 * @param {number} width 单项form（文本和控件）所占百分比1-100
 * @param {number} label 单项form文本宽度（宽度设置的单位为rem）
 * @param {*} other  单项form其他样式名称
 * @example
 * <FormItem className={itemLayout('100', '7')} label={<span>调度名称</span>} field="jobname">
 *   <Input value={form.jobname} />
 * </FormItem>
 */
export const formLayout = (width, label, other) => {
  return classnames(
    `eb-com-form-item-${width}`,
    `eb-com-form-item-label-${label}`,
    other
  );
};
/**
 * form标签每一项的布局
 * @param {number} width 宽度 一般为百分比
 * @param {*} label form表单文字label的宽度单位是em 一般设置为文字的个数
 * @param {*} marginRight form表单左侧的margin宽度 百分比
 * @param {*} display
 */
export const formItemLayout = (width, label, marginRight, display) => {
  return {
    style: {
      width: `${width}%`,
      label: `${label}%`,
      marginRight: `${marginRight || 0}%`,
      display: display ? display : 'inline-block'
    },
    labelWidth: label === '100' ? '100%' : `${label}rem`,
    filled: label !== '100'
  };
};
/**
 * 执行字符串表达式
 * @param {*} fn
 */
const evil = (fn) => {
  let Fn = Function;  // 一个变量指向Function，防止有些前端编译工具报错
  return new Fn(`return ${fn}`)();
};
/**
 * 常用form表单校对象
 */
export const formRules = {
  longTextLength: 1000,
  textLength: 255,
  secondLength: 500,
  hundredLength: 100,
  required: (msg) => ({ required: true, message: msg || '必填项' }),
  name: (length = 32) => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        if (trim(value).length > length) {
          callback(new Error(`字符长度不能大于${length}`));
        }
      }
      callback();
    } };
  },
  equipLength: (length = 32, tip) => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        if (trim(value).length !== length) {
          callback(new Error(tip || `长度必须等于${length}位`));
        }
      }
      callback();
    } };
  },
  numEvilScope: (min, max, tip) => {
    return { validator: (rules, value, callback) => {
      if (value !== undefined && value !== null && trim(String(value))) {
        const _value = Number(trim(String(value)));
        try {
          if (min && evil(`${_value}${min}`) || (max && evil(`${_value}${max}`))) {
            callback(new Error(tip));
          }
        } catch (e) {
          callback(new Error(tip));
        }
      }
      callback();
    } };
  },
  numScope: (min, max, tip) => {
    return { validator: (rules, value, callback) => {
      if (value !== undefined && value !== null && trim(String(value))) {
        if (isNaN(value) || (min !== undefined && min !== null && Number(trim(String(value))) < min) ||
         (max !== undefined && max !== null && Number(trim(String(value))) > max)) {
          callback(new Error(tip));
        }
      }
      callback();
    } };
  },
  phone: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^1(3|4|5|6|7|8|9)\d{9}$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确的手机号`));
        }
      }
      callback();
    } };
  },
  tel: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^((0\d{2,3})-)(\d{7,8})((\d{3,}))?$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确的固定电话`));
        }
      }
      callback();
    } };
  },
  phoneAndtel: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        // const reg = /^1(3|4|5|6|7|8|9)\d{9}$/;
        // const reg1 = /^((0\d{2,3})-)(\d{7,8})((\d{3,}))?$/;
        const reg = /^(^1[3|4|5|6|7|8|9][0-9]\d{4,8}$)|(^(0[0-9]{2,3}\-)?([2-9][0-9]{6,7})+(\-[0-9]{1,4})?$)$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确的手机号或固定电话`));
        }
      }
      callback();
    } };
  },
  serviceTel: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /(^(400)-(\d{3})-(\d{4})(.)(\d{1,4})$)|(^(400)-(\d{3})-(\d{4}$))/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确的客服电话`));
        }
      }
      callback();
    } };
  },
  email: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^\w+([-+.]\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确格式的邮箱地址`));
        }
      }
      callback();
    } };
  },
  website: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^((https|http|ftp|rtsp|mms){0,1}(:\/\/){0,1})www\.(([A-Za-z0-9-~]+)\.)+([A-Za-z0-9-~\/])+$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确格式的网址`));
        }
      }
      callback();
    } };
  },
  longitude: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^(\-|\+)?(((\d|[1-9]\d|1[0-7]\d|0{1,3})\.\d{0,6})|(\d|[1-9]\d|1[0-7]\d|0{1,3})|180\.0{0,6}|180)$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确格式经度`));
        }
      }
      callback();
    } };
  },
  latitude: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^(\-|\+)?([0-8]?\d{1}\.\d{0,6}|90\.0{0,6}|[0-8]?\d{1}|90)$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确格式纬度`));
        }
      }
      callback();
    } };
  },
  clockScope: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^[+]{0,1}(\d+)$|^[+]{0,1}(\d+\.\d+)$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入大于0的数`));
        }
      }
      callback();
    } };
  },
  positiveInteger: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^[1-9]\d*$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正整数`));
        }
      }
      callback();
    } };
  },
  idNumber: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^[1-9]\d{5}(18|19|20)\d{2}((0[1-9])|(1[0-2]))(([0-2][1-9])|10|20|30|31)\d{3}[0-9Xx]$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`请输入正确的身份证号！`));
        }
      }
      callback();
    } };
  },
  internalSpace: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /\s/;
        if (reg.test(trim(value))) {
          callback(new Error(`字符串内部不能有空格`));
        }
      }
      callback();
    } };
  },
  isInteger: () => {
    return { validator: (rules, value, callback) => {
      if (!Number.isInteger(value)) {
        callback(new Error(`请输入整数`));
      }
      callback();
    } };
  },

  // 校验空格
  spaceCheck: () => {
    return { validator: (rules, value, callback) => {
      if (value && trim(value)) {
        const reg = /^[^ ]+$/;
        if (!reg.test(trim(value))) {
          callback(new Error(`字符串内部不能有空格`));
        }
      }
      callback();
    } };
  }

};

/**
 * 过滤对象中字符串的前后空格
 * @param {*} form
 */
export const trimFormData = (form) => {
  for (let key in form) {
    if ((typeof form[key] === 'string') && form[key].constructor === String) {
      form[key] = trim(form[key]);
    }
  }
  return form;
};
