package com.yeejoin.amos.api.tool.utils;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

public class DateUtils {
    public static String LONG_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public static String MID_PATTERN = "yyyy-MM-dd HH:mm";
    public static String SHORT_PATTERN = "yyyy-MM-dd";
    public static String MONTH_PATTERN = "yyyy-MM";

    private static final SimpleDateFormat monthSdf = new SimpleDateFormat("yyyy-MM");
    private static final SimpleDateFormat shortSdf = new SimpleDateFormat("yyyy-MM-dd");
    private static final SimpleDateFormat longSdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    public static Date parseDate(String date) {
        try {
            return new SimpleDateFormat("yyyy-MM-dd hh:mm:ss").parse(date);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static String toDateStr(Date date) {
        return new SimpleDateFormat("yyyy-MM-dd hh:mm:ss").format(date);
    }

    public static Date addDate(Date date, int field, int amount) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.add(field, amount);
        return c.getTime();
    }

    public static boolean isValidDate(String str, String format) {
        DateFormat formatter = new SimpleDateFormat(format);
        try {
            Date date = (Date) formatter.parse(str);
            return str.equals(formatter.format(date));
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 获取某年某月的第一天日期
     *
     * @param date
     * @param format
     * @return
     */
    public static String getStartMonthDate(String date, String format) {
        if (date == null || date.length() < 6 || format == null) {
            return null;
        }
        int year = Integer.parseInt(date.substring(0, 4));
        int month = date.contains("-") ? month = Integer.parseInt(date.substring(5, 7)) : Integer.parseInt(date.substring(4, 6));
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, 1);
        return shortSdf.format(cal.getTime()) + " 00:00:00";
    }

    /**
     * 获取某年某月的最后一天日期
     *
     * @param date
     * @param format
     * @return
     */
    public static String getEndMonthDate(String date, String format) {
        if (date == null || date.length() < 6 || format == null) {
            return null;
        }
        int year = Integer.parseInt(date.substring(0, 4));
        int month = date.contains("-") ? month = Integer.parseInt(date.substring(5, 7)) : Integer.parseInt(date.substring(4, 6));
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, 1);
        int day = cal.getActualMaximum(5);
        cal.set(year, month - 1, day);
        return shortSdf.format(cal.getTime()) + " 23:59:59";
    }

    /**
     * 获取某天的开始时间
     *
     * @param date
     * @param format
     * @return
     */
    public static String getStartDayDate(String date, String format) {
        try {
            return getCurrentDayStartTime(new SimpleDateFormat(SHORT_PATTERN).parse(date));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取某天的结束时间
     *
     * @param date
     * @param format
     * @return
     */
    public static String getEndDayDate(String date, String format) {
        try {
            return getCurrentDayEndTime(new SimpleDateFormat(SHORT_PATTERN).parse(date));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获得本天的开始时间，即2012-01-01 00:00:00
     *
     * @return
     */
    public static String getCurrentDayStartTime(Date date) throws Exception {
        date = shortSdf.parse(shortSdf.format(date) + " 00:00:00");
        return longSdf.format(date);
    }

    /**
     * 获得本天的结束时间，即2012-01-01 23:59:59
     *
     * @return
     */
    public static String getCurrentDayEndTime(Date date) throws Exception {
        date = longSdf.parse(shortSdf.format(date) + " 23:59:59");
        return longSdf.format(date);
    }

    /**
     * 获得本周的第一天，周一
     *
     * @return
     */
    public static String getCurrentWeekDayStartTime(Date date) throws Exception {
        Calendar c = Calendar.getInstance();
        Date dt = null;
        c.setTime(date);
        int weekday = c.get(Calendar.DAY_OF_WEEK) - 2;
        c.add(Calendar.DATE, -weekday);
        dt = longSdf.parse(shortSdf.format(c.getTime()) + " 00:00:00");
        return longSdf.format(dt);
    }

    /**
     * 获得本周的最后一天，周日
     *
     * @return
     */
    public static String getCurrentWeekDayEndTime(Date date) throws Exception {
        Calendar c = Calendar.getInstance();
        Date dt = null;
        c.setTime(date);
        int weekday = c.get(Calendar.DAY_OF_WEEK);
        c.add(Calendar.DATE, 8 - weekday);
        dt = longSdf.parse(shortSdf.format(c.getTime()) + " 23:59:59");
        return longSdf.format(dt);
    }

    /**
     * 获得本月的开始时间，即2012-01-01 00:00:00
     *
     * @return
     */
    public static String getCurrentMonthStartTime(Date date) throws Exception {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        Date dt = null;
        c.set(Calendar.DATE, 1);
        dt = shortSdf.parse(shortSdf.format(c.getTime()) + " 00:00:00");
        return longSdf.format(dt);
    }

    /**
     * 当前月的结束时间，即2012-01-31 23:59:59
     *
     * @return
     */
    public static String getCurrentMonthEndTime(Date date) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        Date now = null;
        c.set(Calendar.DATE, 1);
        c.add(Calendar.MONTH, 1);
        c.add(Calendar.DATE, -1);
        try {
            now = longSdf.parse(shortSdf.format(c.getTime()) + " 23:59:59");
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return longSdf.format(now);
    }

    /**
     * 当前季度的开始时间，即2012-01-1 00:00:00
     *
     * @return
     */
    public static String getCurrentQuarterStartTime(Date date) throws Exception {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        int currentMonth = c.get(Calendar.MONTH) + 1;
        Date dt = null;
        if (currentMonth >= 1 && currentMonth <= 3)
            c.set(Calendar.MONTH, 0);
        else if (currentMonth >= 4 && currentMonth <= 6)
            c.set(Calendar.MONTH, 3);
        else if (currentMonth >= 7 && currentMonth <= 9)
            c.set(Calendar.MONTH, 4);
        else if (currentMonth >= 10 && currentMonth <= 12)
            c.set(Calendar.MONTH, 9);
        c.set(Calendar.DATE, 1);
        dt = longSdf.parse(shortSdf.format(c.getTime()) + " 00:00:00");
        return longSdf.format(dt);
    }

    /**
     * 当前季度的结束时间，即2012-03-31 23:59:59
     *
     * @return
     */
    public static String getCurrentQuarterEndTime(Date date) throws Exception {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        int currentMonth = c.get(Calendar.MONTH) + 1;
        Date dt = null;
        if (currentMonth >= 1 && currentMonth <= 3) {
            c.set(Calendar.MONTH, 2);
            c.set(Calendar.DATE, 31);
        } else if (currentMonth >= 4 && currentMonth <= 6) {
            c.set(Calendar.MONTH, 5);
            c.set(Calendar.DATE, 30);
        } else if (currentMonth >= 7 && currentMonth <= 9) {
            c.set(Calendar.MONTH, 8);
            c.set(Calendar.DATE, 30);
        } else if (currentMonth >= 10 && currentMonth <= 12) {
            c.set(Calendar.MONTH, 11);
            c.set(Calendar.DATE, 31);
        }
        dt = longSdf.parse(shortSdf.format(c.getTime()) + " 23:59:59");
        return longSdf.format(dt);
    }

    /**
     * 当前年的开始时间，即2012-01-01 00:00:00
     *
     * @return
     */
    public static String getCurrentYearStartTime(Date date) throws Exception {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        Date dt = null;
        c.set(Calendar.MONTH, 0);
        c.set(Calendar.DATE, 1);
        dt = shortSdf.parse(shortSdf.format(c.getTime()) + " 00:00:00");
        return longSdf.format(dt);
    }

    /**
     * 当前年的结束时间，即2012-12-31 23:59:59
     *
     * @return
     */
    public static String getCurrentYearEndTime(Date date) throws Exception {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        Date dt = null;
        c.set(Calendar.MONTH, 11);
        c.set(Calendar.DATE, 31);
        dt = longSdf.parse(shortSdf.format(c.getTime()) + " 23:59:59");
        return longSdf.format(dt);
    }

    /**
     * 获取当前时间
     *
     * @return
     */
    public static Date format() {
        Date date = new Date();
        return date;
    }

    /**
     * 获取前一天日期
     *
     * @return
     */
    public static Date getYesterday() {
        Calendar cal = Calendar.getInstance();
        cal.setTime(format());
        cal.add(Calendar.DATE, -1);
        return cal.getTime();
    }

    /**
     * 获取过去七天
     *
     * @return
     */
    public static Date getBeforeWeek(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(format());
        cal.add(Calendar.DATE, -7);
        Date beforeWeek = cal.getTime();
        return beforeWeek;
    }

    /**
     * 获取过去一月
     *
     * @return
     */
    public static Date getBeforeMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(format());
        cal.add(Calendar.MONTH, -1);
        Date beforeWeek = cal.getTime();
        return beforeWeek;
    }

    /**
     * 获取过去三个月
     *
     * @return
     */
    public static Date getBeforeQuarter(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(format());
        cal.add(Calendar.MONTH, -3);
        Date beforeQuarter = cal.getTime();
        return beforeQuarter;
    }

    /**
     * 获取去年今日
     *
     * @return
     */
    public static Date getBeforeYear(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.YEAR, -1);
        Date y = cal.getTime();
        String year = shortSdf.format(y);
        return y;
    }
}
